package internal

import (
	"log"
	"os"

	"code.justin.tv/common/config"

	"github.com/cactus/go-statsd-client/statsd"
)

func init() {
	config.Register(map[string]string{
		"datadog-agent-host-port":  "127.0.0.1:8125",
		"signalfx-agent-host-port": "127.0.0.1:8135",
	})
}

// StatsEndpoint defines possible endpoints for metrics
type StatsEndpoint string

const (
	// EndpointStatsd our internal Twitch statsd service
	EndpointStatsd = StatsEndpoint("statsd")
	// EndpointDataDog datadog agent
	EndpointDataDog = StatsEndpoint("datadog")
	// EndpointSignalFX signalfx agent
	EndpointSignalFX = StatsEndpoint("signalfx")
)

// MultiStatter is the struct holding the underlying cactus statter as well
// as the sender for multiple StatsEndpoints
type MultiStatter struct {
	statsd.Statter
	multiSender multiSender
}

// New creates a new MultiStatter that implements the necessary logic to
// send payloads to two different StatsEndpoints (statsd default service  and local
// datadog agent) using two different protocols (statsd and dogstatsd respectively)
func New(endpoints ...StatsEndpoint) (*MultiStatter, error) {
	if len(endpoints) == 0 {
		endpoints = []StatsEndpoint{
			EndpointStatsd,
			EndpointDataDog,
			EndpointSignalFX,
		}
	}
	senders := map[StatsEndpoint]statsd.Sender{}

	for _, endpoint := range endpoints {
		var sender statsd.Sender
		var err error
		switch endpoint {
		case EndpointStatsd:
			sender, err = newStatsdSender(config.StatsdHostPort())
			break
		case EndpointDataDog:
			sender, err = newDogstatsdSender(config.Resolve("datadog-agent-host-port"))
			break
		case EndpointSignalFX:
			sender, err = newDogstatsdSender(config.Resolve("signalfx-agent-host-port"))
			break
		}
		if err != nil {
			return nil, err
		}
		senders[endpoint] = sender
	}

	multisender := multiSender{
		senders: senders,
	}

	statter, err := statsd.NewClientWithSender(multisender, "")
	if err != nil {
		return nil, err
	}

	return &MultiStatter{
		Statter:     statter,
		multiSender: multisender,
	}, nil
}

// WithEndpointsOnly returns a copy of this multiSender that can only forward
// payload to the specified endpoints (filters out the rest)
func (s *MultiStatter) WithEndpointsOnly(endpoints ...StatsEndpoint) *MultiStatter {
	if s == nil {
		return nil
	}

	newSenders := map[StatsEndpoint]statsd.Sender{}
	for _, endpoint := range endpoints {
		newSenders[endpoint] = s.multiSender.senders[endpoint]
	}

	newMultiSender := multiSender{senders: newSenders}

	// Discard error as it is only a check for sender != nil
	newStatter, _ := statsd.NewClientWithSender(newMultiSender, "")

	return &MultiStatter{
		Statter:     newStatter,
		multiSender: newMultiSender,
	}
}

// WithTags returns a new multistatter and adds tags to any underlying sender
// that supports them. Analogous to the NewSubStatter method on the original statter
func (s *MultiStatter) WithTags(tags map[string]string) *MultiStatter {
	if s == nil {
		return nil
	}

	newSenders := map[StatsEndpoint]statsd.Sender{}
	for endpoint, sender := range s.multiSender.senders {
		tagsSender, ok := (sender).(dogstatsdSender)
		if ok {
			newSenders[endpoint] = tagsSender.withTags(tags)
			continue
		}
		newSenders[endpoint] = sender
	}

	newSender := multiSender{senders: newSenders}

	// Discard error as it is only a check for sender != nil
	newStatter, _ := statsd.NewClientWithSender(newSender, "")

	return &MultiStatter{
		Statter:     newStatter,
		multiSender: newSender,
	}
}

// AsStatsdStatter takes a MultiStatter and returns it as a cactus statsd
// compatible statter type.
func AsStatsdStatter(s *MultiStatter) statsd.Statter {
	statsdStatter := statsd.Statter(*s)
	return statsdStatter
}

// AsMultiStatter takes a cactus statsd statter and converts it to MultiStatter
// if the underlying type is correct. nil otherwise
func AsMultiStatter(s statsd.Statter) *MultiStatter {
	multiStatter, ok := (s).(MultiStatter)
	if !ok {
		return nil
	}
	return &multiStatter
}

type multiSender struct {
	senders map[StatsEndpoint]statsd.Sender
}

func (s multiSender) Send(data []byte) (int, error) {
	for endpoint, sender := range s.senders {
		if _, err := sender.Send(data); err != nil {
			log.Printf("An error occurred sending metrics to the %s service: %#+v", string(endpoint), err)
		}
	}

	// number of bytes written is ignored on cacturs/go-statsd-client implementation
	// so we ignore it here too.
	return 0, nil
}

func (s multiSender) Close() error {
	for StatsEndpoint, sender := range s.senders {
		if err := sender.Close(); err != nil {
			log.Printf("An error occurred while closing the connection to the %s metrics StatsEndpoint: %#+v", string(StatsEndpoint), err)
		}
	}
	return nil
}

func sanitizedHostname() string {
	hostname, err := os.Hostname()
	if err != nil {
		return "unknown"
	}

	return Sanitize(hostname)
}
