package clips

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/foundation/twitchclient"
)

type ClientError struct {
	Code    string `json:"code"`
	Message string `json:"message"`
}

// Error returns the code and message as a string.
func (e *ClientError) Error() string {
	return fmt.Sprintf("%s: %s", e.Code, e.Message)
}

// NewClientError returns a new ClientError that contains the error code and message
// from the Clips API. If there is no response, the body is invalid, or no code is
// provided in the body, no client error is returned and the twitchclient error should
// be returned.
func NewClientError(inputErr error) *ClientError {
	if inputErr == nil {
		return nil
	}

	if twitchErr, ok := inputErr.(*twitchclient.Error); ok {
		var clientErr ClientError
		err := json.Unmarshal([]byte(twitchErr.Message), &clientErr)
		if err != nil {
			return nil
		}
		if clientErr.Code == "" {
			return nil
		}

		return &clientErr
	}
	return nil
}
