package flipper

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/url"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/flipper/models"
)

const (
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "flipper"
)

// Client is an interface that exposes methods to fetch data from the flipper service.
type Client interface {
	// GetChannelIDByCommentID returns the ChannelID that the comment was published on.
	GetChannelIDByCommentID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (string, error)

	// GetCommentByID returns a comment given the commentID.
	GetCommentByID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// CreateComment will create a new comment given a valid ContentType and ContentID. Currently only ContentType "video" is supported.
	CreateComment(ctx context.Context, commentProperties *models.CommentUpdate, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// UpdateComment updates a comment given the commentID and commentUpdate with the desired updated fields.
	// Currently can only update the comment's State.
	UpdateComment(ctx context.Context, commentID string, commentUpdate *models.CommentUpdate, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// DeleteComment will delete a comment given the commentID.
	DeleteComment(ctx context.Context, commentID string, commentDelete *models.CommentDeleteInput, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// GetCommentsByVOD will return a CommentsResult that contains the top-level comments on the VOD around
	// the offset provided with a default of commentsByVODLimitTopLevelDefault, up to a limit of commentsByVODLimitTopLevelMax top-level comments.
	GetCommentsByVOD(ctx context.Context, vodID string, offset *time.Duration, commentID *string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error)

	// GetCommentsByParentID returns all child comments of a comment in reverse chronological order along with a cursor that
	// allows retrieving additional comments.
	GetCommentsByParentID(ctx context.Context, parentID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error)

	// GetModerationCapabilitiesByChannelID returns capabilities of the authenticated user for the channel specified
	GetModerationCapabilitiesByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*models.ModerationCapabilities, error)

	// GetModerationCommentsByChannelID returns all comments of a channel in reverse chronological order along with a cursor that
	// allows retrieving additional comments.
	GetModerationCommentsByChannelID(ctx context.Context, channelID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error)

	// GetModerationCommentsByVOD returns all comments of a VOD in reverse chronological order along with a cursor that
	// allows retrieving additional comments.
	GetModerationCommentsByVOD(ctx context.Context, vodID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error)

	// GetModerationSettingsByChannelID returns all flipper settings for the specified channelID.
	GetModerationSettingsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*models.Settings, error)

	// UpdateModerationSettingsByChannelID updates the specified channelID's flipper settings given a SettingsUpdate and returns the updated settings
	UpdateModerationSettingsByChannelID(ctx context.Context, channelID string, settingsUpdate *models.SettingsUpdate, reqOpts *twitchclient.ReqOpts) (*models.Settings, error)

	// CreateCommentReport creates a new report for a comment.
	CreateCommentReport(ctx context.Context, commentReport *models.CommentReport, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// CreateChannelBan bans a commenter from a channel and deletes all comments created by that user on the channel.
	CreateChannelBan(ctx context.Context, channelBan *models.ChannelBan, reqOpts *twitchclient.ReqOpts) error

	// InternalGetCommentByID returns a comment given the commentID.
	InternalGetCommentByID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (*models.Comment, error)

	// InternalGetCommentsByVOD will return a CommentsResult that contains the top-level comments on the VOD around the offset provided
	// with a default of commentsByVODLimitTopLevelDefault, up to a limit of commentsByVODLimitTopLevelMax top-level comments.
	InternalGetCommentsByVOD(ctx context.Context, vodID string, offset *time.Duration, commentID *string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error)
}

type clientImpl struct {
	twitchclient.Client
}

// NewClient creates a client for the flipper service.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{twitchClient}, nil
}

func (c *clientImpl) GetChannelIDByCommentID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (string, error) {
	path := fmt.Sprintf("/comments/%s/channel", url.QueryEscape(commentID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return "", err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_channel",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	_, err = c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return "", err
	}

	return comment.ChannelID, nil
}

func (c *clientImpl) GetCommentByID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	path := fmt.Sprintf("/comments/%s", url.QueryEscape(commentID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_comment",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	_, err = c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &comment, nil
}

func (c *clientImpl) CreateComment(ctx context.Context, commentProperties *models.CommentUpdate, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	input, err := json.Marshal(commentProperties)
	if err != nil {
		return nil, err
	}
	req, err := c.NewRequest("POST", "/comments", bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.create_comment",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	_, err = c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &comment, nil
}

func (c *clientImpl) UpdateComment(ctx context.Context, commentID string, commentUpdate *models.CommentUpdate, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	input, err := json.Marshal(commentUpdate)
	if err != nil {
		return nil, err
	}

	path := fmt.Sprintf("/moderation/comments/%s", url.QueryEscape(commentID))
	req, err := c.NewRequest("PATCH", path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.update_comment",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	_, err = c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &comment, nil
}

func (c *clientImpl) DeleteComment(ctx context.Context, commentID string, commentDeleteInput *models.CommentDeleteInput, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	input, err := json.Marshal(commentDeleteInput)
	if err != nil {
		return nil, err
	}

	path := fmt.Sprintf("/moderation/comments/%s", url.QueryEscape(commentID))
	req, err := c.NewRequest("DELETE", path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.delete_comment",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	resp, err := c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode >= 400 {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &comment, nil
}

func (c *clientImpl) GetCommentsByVOD(ctx context.Context, vodID string, offset *time.Duration, commentID *string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error) {
	path := fmt.Sprintf("/videos/%s/comments", url.QueryEscape(vodID))

	params := url.Values{}
	if offset != nil {
		params.Add("offset", offset.String())
	}
	if cursor != nil {
		params.Add("cursor", *cursor)
	}
	if commentID != nil {
		params.Add("comment_id", *commentID)
	}
	if limit != nil {
		params.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if len(params) > 0 {
		path = fmt.Sprintf("%s?%s", path, params.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_comments_by_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	comments := &models.CommentsResult{}
	_, err = c.DoJSON(ctx, &comments, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return comments, nil
}

func (c *clientImpl) GetCommentsByParentID(ctx context.Context, parentID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error) {
	path := fmt.Sprintf("/comments/%s/children", url.QueryEscape(parentID))
	params := url.Values{}
	if cursor != nil {
		params.Add("cursor", *cursor)
	}
	if limit != nil {
		params.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if len(params) > 0 {
		path = fmt.Sprintf("%s?%s", path, params.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_comments_by_parent_id",
		StatSampleRate: defaultStatSampleRate,
	})

	comments := &models.CommentsResult{}
	_, err = c.DoJSON(ctx, &comments, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return comments, nil

}

func (c *clientImpl) GetModerationCapabilitiesByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*models.ModerationCapabilities, error) {
	path := fmt.Sprintf("/moderation/channels/%s/capabilities", url.QueryEscape(channelID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_moderation_capabilities_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	capabilities := &models.ModerationCapabilities{}
	_, err = c.DoJSON(ctx, &capabilities, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return capabilities, nil
}

func (c *clientImpl) GetModerationCommentsByChannelID(ctx context.Context, channelID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error) {
	path := fmt.Sprintf("/moderation/channels/%s/comments", url.QueryEscape(channelID))
	params := url.Values{}
	if cursor != nil {
		params.Add("cursor", *cursor)
	}
	if limit != nil {
		params.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if len(params) > 0 {
		path = fmt.Sprintf("%s?%s", path, params.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_moderation_comments_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	comments := &models.CommentsResult{}
	_, err = c.DoJSON(ctx, &comments, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return comments, nil
}

func (c *clientImpl) GetModerationCommentsByVOD(ctx context.Context, vodID string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error) {
	path := fmt.Sprintf("/moderation/videos/%s/comments", url.QueryEscape(vodID))
	params := url.Values{}
	if cursor != nil {
		params.Add("cursor", *cursor)
	}
	if limit != nil {
		params.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if len(params) > 0 {
		path = fmt.Sprintf("%s?%s", path, params.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_moderation_comments_by_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	comments := &models.CommentsResult{}
	_, err = c.DoJSON(ctx, &comments, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return comments, nil
}

func (c *clientImpl) GetModerationSettingsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*models.Settings, error) {
	path := fmt.Sprintf("/moderation/channels/%s/settings", url.QueryEscape(channelID))

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.get_moderation_settings_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	settings := &models.Settings{}
	_, err = c.DoJSON(ctx, &settings, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return settings, nil
}

func (c *clientImpl) UpdateModerationSettingsByChannelID(ctx context.Context, channelID string, settingsUpdate *models.SettingsUpdate, reqOpts *twitchclient.ReqOpts) (*models.Settings, error) {
	input, err := json.Marshal(settingsUpdate)
	if err != nil {
		return nil, err
	}

	path := fmt.Sprintf("/moderation/channels/%s/settings", url.QueryEscape(channelID))

	req, err := c.NewRequest("PATCH", path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.update_moderation_settings_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	settings := &models.Settings{}
	_, err = c.DoJSON(ctx, &settings, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return settings, nil
}

func (c *clientImpl) CreateCommentReport(ctx context.Context, commentReport *models.CommentReport, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	input, err := json.Marshal(commentReport)
	if err != nil {
		return nil, err
	}

	path := fmt.Sprintf("/comments/%s/reports", url.QueryEscape(commentReport.CommentID))

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.create_comment_report",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	resp, err := c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode >= 400 {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &comment, nil
}

func (c *clientImpl) CreateChannelBan(ctx context.Context, channelBan *models.ChannelBan, reqOpts *twitchclient.ReqOpts) error {
	input, err := json.Marshal(channelBan)
	if err != nil {
		return err
	}

	path := fmt.Sprintf("/moderation/channels/%s/bans", url.QueryEscape(channelBan.ChannelID))

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(input))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.create_channel_ban",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) InternalGetCommentByID(ctx context.Context, commentID string, reqOpts *twitchclient.ReqOpts) (*models.Comment, error) {
	path := fmt.Sprintf("/internal/comments/%s", url.QueryEscape(commentID))
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.internal_get_comment",
		StatSampleRate: defaultStatSampleRate,
	})

	comment := models.Comment{}
	_, err = c.DoJSON(ctx, &comment, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &comment, nil
}

func (c *clientImpl) InternalGetCommentsByVOD(ctx context.Context, vodID string, offset *time.Duration, commentID *string, cursor *string, limit *int64, reqOpts *twitchclient.ReqOpts) (*models.CommentsResult, error) {
	path := fmt.Sprintf("/internal/videos/%s/comments", url.QueryEscape(vodID))

	params := url.Values{}
	if offset != nil {
		params.Add("offset", offset.String())
	}
	if cursor != nil {
		params.Add("cursor", *cursor)
	}
	if commentID != nil {
		params.Add("comment_id", *commentID)
	}
	if limit != nil {
		params.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if len(params) > 0 {
		path = fmt.Sprintf("%s?%s", path, params.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.flipper.internal_get_comments_by_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	comments := &models.CommentsResult{}
	_, err = c.DoJSON(ctx, &comments, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return comments, nil
}
