package models

import (
	"fmt"
	"time"
)

// CommentState describes the visibility state of a comment
type CommentState string

// CommentSource describes where the comment originated
type CommentSource string

// CommentsSortField describes the sort option that should be applied to a query
type CommentsSortField string

const (
	// CommentStatePublished indicates that the comment is published
	CommentStatePublished CommentState = "published"
	// CommentStateUnpublished indicates that the comment was manually unpublished by a mod or the channel owner
	CommentStateUnpublished CommentState = "unpublished"
	// CommentStatePendingReview indicates that the comment was automatically flagged for review (AutoMod)
	CommentStatePendingReview CommentState = "pending_review"
	// CommentStatePendingReviewSpam indicates that the comment is possibly spam and has to be reviewed by a moderator
	CommentStatePendingReviewSpam CommentState = "pending_review_spam"

	// CommentSourceChompy indicates that the comment was manually created by a commenter
	CommentSourceChompy CommentSource = "chompy"
	// CommentSourceRechat indicates that the comment was recorded during a live broadcast
	CommentSourceRechat CommentSource = "rechat"

	// CommentsSortCreatedAt indicates that the request should be sorted by the CreatedAt field
	CommentsSortCreatedAt = "created_at"
	// CommentsSortContentID indicates that the request should be sorted by the ContentID field
	CommentsSortContentID = "content_id"
	// CommentsSortChannelID indicates that the request should be sorted by the ChannelID field
	CommentsSortChannelID = "channel_id"
	// CommentsSortContentOffset indicates that the request should be sorted by the ContentOffset field
	CommentsSortContentOffset = "content_offset"
)

var (
	validCommentStates = map[CommentState]struct{}{
		CommentStatePublished:         {},
		CommentStateUnpublished:       {},
		CommentStatePendingReview:     {},
		CommentStatePendingReviewSpam: {},
	}
	validCommentSources = map[CommentSource]struct{}{
		CommentSourceChompy: {},
		CommentSourceRechat: {},
	}
	validCommentsSortFields = map[CommentsSortField]struct{}{
		CommentsSortCreatedAt: {},
		CommentsSortContentID: {},
		CommentsSortChannelID: {},
	}
)

// Comment is ... a comment OpieOP
type Comment struct {
	ID            string          `json:"id"`
	CreatedAt     time.Time       `json:"created_at"`
	UpdatedAt     time.Time       `json:"updated_at"`
	PublishedAt   *time.Time      `json:"published_at,omitempty"`
	ParentID      *string         `json:"parent_id"`
	ChannelID     string          `json:"channel_id"`
	ContentObject *ContentObject  `json:"content"`
	ContentOffset *ContentOffset  `json:"content_offset"`
	CommenterID   string          `json:"commenter_id"`
	State         CommentState    `json:"state"`
	Deleted       bool            `json:"deleted"`
	Message       *CommentMessage `json:"message"`
	ModifiedByID  *string         `json:"modified_by_id,omitempty"`
	Source        CommentSource   `json:"source"`
	ScanReplies   *bool           `json:"scan_replies,omitempty"`

	// dynamically loaded
	Children           CommentList `json:"children,omitempty"`
	MoreChildren       *bool       `json:"more_children,omitempty"`
	MoreChildrenCursor *string     `json:"more_children_cursor,omitempty"`
	Reactions          []*Reaction `json:"reactions,omitempty"`
	UserReactions      []string    `json:"user_reactions,omitempty"`
}

// CommentList is a list of comments
type CommentList []*Comment

// CommentListByCreatedAt is a list of comment that can be sorted by creation date
type CommentListByCreatedAt CommentList

func (l CommentListByCreatedAt) Len() int {
	return len(l)
}

func (l CommentListByCreatedAt) Swap(i, j int) {
	l[i], l[j] = l[j], l[i]
}

func (l CommentListByCreatedAt) Less(i, j int) bool {
	return l[i].CreatedAt.Before(l[j].CreatedAt)
}

// CommentUpdate describes how an existing comment should be changed
type CommentUpdate struct {
	Message       *string        `json:"message"`
	ParentID      *string        `json:"parent_id"`
	ContentType   *string        `json:"content_type"`
	ContentID     *string        `json:"content_id"`
	ContentOffset *ContentOffset `json:"content_offset"`
	State         *CommentState  `json:"state"`
	UpdatedAt     *time.Time     `json:"updated_at,omitempty"`
	PublishedAt   *time.Time     `json:"published_at,omitempty"`
	ModifiedByID  *string        `json:"modified_by_id,omitempty"`
	Deleted       *bool          `json:"deleted,omitempty"`
	ScanReplies   *bool          `json:"scan_replies,omitempty"`
}

// CommentDeleteInput has fields required for comment deletion
type CommentDeleteInput struct {
	UserID string `json:"user_id"`
}

// CommentsInOffsetRangeResult is a list of comments and an offset range that describes which comments are included
type CommentsInOffsetRangeResult struct {
	Comments           CommentList         `json:"comments"`
	ContentOffsetRange *ContentOffsetRange `json:"offset_range"`
}

// CommentsResult is a set of comments plus cursors to fetch previous/next results
type CommentsResult struct {
	Comments       CommentList `json:"comments"`
	Parents        CommentList `json:"parents,omitempty"`
	PreviousCursor *string     `json:"previous_cursor,omitempty"`
	NextCursor     *string     `json:"next_cursor,omitempty"`
}

// CommentsFilter is a set of filters that can be applied on comments to filter out
// certain types of comments.
type CommentsFilter struct {
	ExcludeUnpublished bool `json:"exclude_unpublished"`
}

// CacheKey returns a string representation of the data in the filter that can be used as part of a cache key
func (f *CommentsFilter) CacheKey() string {
	return fmt.Sprintf("excl_unpub:%t", f.ExcludeUnpublished)
}

// CommentsOffset is a set of offset fields used to specify a query request
type CommentsOffset struct {
	OffsetBegin *ContentOffset `json:"offset"`
	// For subcontent i.e. highlights, clips
	OffsetRange *ContentOffsetRange `json:"offset_range"`
}

// CommentsSort is a set of sorting options available to specify on some
// queries (ES queries). If a CommentsSort is not specified, the default behavior is
// to sort by CreatedAt, Descending
type CommentsSort struct {
	SortField string `json:"sort_field"`
	Ascending bool   `json:"ascending"`
}

// CommentMessage is the message of a comment
type CommentMessage struct {
	Body       string           `json:"body"`
	BitsSpent  *int             `json:"bits_spent,omitempty"`
	Emoticons  []*EmoticonRange `json:"emoticons,omitempty"`
	IsAction   bool             `json:"action"`
	UserBadges []*UserBadge     `json:"user_badges,omitempty"`
	UserColor  *string          `json:"user_color,omitempty"`

	// If present, this is the parsed version of the comment body. Assembling
	// the Text of each fragment in order should be identical to the Body.
	// Use Fragments for things like mentions and emoticons.
	Fragments []*CommentMessageFragment `json:"fragments,omitempty"`

	// Risk
	AutoModFlagged bool    `json:"automod_flagged"`
	SpamLikelihood float64 `json:"spam_likelihood"`
}

// EmoticonRange describes where in the body an emoticon should appear.
type EmoticonRange struct {
	ID    string `json:"id"`
	Begin int    `json:"begin"`
	End   int    `json:"end"`
}

// CommentMessageFragment represents a structured view of a part of comment
// so clients can understand special fields without their own parsing.
type CommentMessageFragment struct {
	Text string `json:"text"`

	Emoticon *CommentMessageFragmentEmoticon `json:"emoticon,omitempty"`
	Mention  *CommentMessageFragmentMention  `json:"mention,omitempty"`
}

// CommentMessageFragmentEmoticon represents an emoticon that appears in a
// comment fragment.
type CommentMessageFragmentEmoticon struct {
	EmoticonID    string `json:"id"`
	EmoticonSetID string `json:"set_id"`
}

// CommentMessageFragmentMention represents a mention of a user that appears
// in the body of a message.
type CommentMessageFragmentMention struct {
	UserID      string `json:"user_id"`
	Login       string `json:"login"`
	DisplayName string `json:"display_name"`
}

// Reaction is a summary of reactions to a comment
type Reaction struct {
	ID    string `json:"id"`
	Count int    `json:"count"`
}

// UserBadge describes a chat badge of a user
type UserBadge struct {
	ID      string `json:"id"`
	Version string `json:"version"`
}

// IsValidCommentState returns true if commentState describes a valid comment state
func IsValidCommentState(commentState string) bool {
	_, ok := validCommentStates[CommentState(commentState)]
	return ok
}

// IsValidCommentSource returns true if commentSource describes a valid comment source
func IsValidCommentSource(commentSource string) bool {
	_, ok := validCommentSources[CommentSource(commentSource)]
	return ok
}

// IsValidCommentsSortField returns true if commentSort describes a valid comment sort option
func IsValidCommentsSortField(commentsSortField string) bool {
	_, ok := validCommentsSortFields[CommentsSortField(commentsSortField)]
	return ok
}

// IsPublished returns true if the comment is published and should be publicly visible
func (c *Comment) IsPublished() bool {
	return c.State == CommentStatePublished
}

// IsChild returns true if the comment is the child of another comment
func (c *Comment) IsChild() bool {
	return c.ParentID != nil
}

// IsRechat return true if the comment was created from live chat
func (c *Comment) IsRechat() bool {
	return c.Source == CommentSourceRechat
}
