package models

import (
	"fmt"
	"time"
)

const (
	// ContentTypeBroadcast represents a past broadcast
	ContentTypeBroadcast = "broadcast"

	// ContentTypeVideo represents an uploaded video
	ContentTypeVideo = "video"
)

// ContentOffsetRange describes the offset range of comments
type ContentOffsetRange struct {
	// Begin is the beginning of the offset range (inclusive)
	Begin *ContentOffset `json:"begin"`
	// End marks the end of the offset range (exclusive)
	End *ContentOffset `json:"end"`
}

// ContentObject describes the content type and related id
type ContentObject struct {
	Type string `json:"type"`
	ID   string `json:"id"`
}

// ContentOffset is the Duration from the start of the content
type ContentOffset struct {
	*Duration
}

// NewContentOffset creates a new ContentOffset given a duration
func NewContentOffset(duration time.Duration) *ContentOffset {
	return &ContentOffset{
		Duration: NewDuration(duration),
	}
}

// Floor returns a *ContentOffset rounded down to the nearest second
func (o *ContentOffset) Floor() *ContentOffset {
	return &ContentOffset{
		Duration: NewDuration(time.Duration(o.Seconds()) * time.Second),
	}
}

// Truncate returns a *ContentOffset rounded toward zero to a multiple of m.
func (o *ContentOffset) Truncate(m time.Duration) *ContentOffset {
	return &ContentOffset{
		Duration: NewDuration(time.Duration(o.Nanoseconds() - (o.Nanoseconds() % uint64(m.Nanoseconds())))),
	}
}

// TimestampString returns the offset as Twitch Timestamp String™ (e.g. 21h42m23s).
// This is the format supported by the video player.
func (o *ContentOffset) TimestampString() string {
	seconds := int64(o.Seconds())
	var hours, minutes int64
	hours = seconds / 60 / 60
	seconds = seconds - hours*60*60
	minutes = seconds / 60
	seconds = seconds - minutes*60
	return fmt.Sprintf("%02dh%02dm%02ds", hours, minutes, seconds)
}

// UnmarshalJSON unmarshals the json representation of a ContentOffset
func (o *ContentOffset) UnmarshalJSON(b []byte) error {
	o.Duration = &Duration{}
	return o.Duration.UnmarshalJSON(b)
}
