package models

import (
	"encoding/json"
	"fmt"
	"time"
)

// Duration is wrapper around time.Duration
type Duration struct {
	duration time.Duration
}

// NewDuration creates a new Duration
func NewDuration(duration time.Duration) *Duration {
	return &Duration{duration: duration}
}

// MarshalJSON marshals a Duration into json
func (d *Duration) MarshalJSON() ([]byte, error) {
	s := fmt.Sprintf("\"%s\"", d.String())
	return []byte(s), nil
}

// UnmarshalJSON unmarshals the json representation of a Duration
func (d *Duration) UnmarshalJSON(b []byte) error {
	var s string
	err := json.Unmarshal(b, &s)
	if err != nil {
		return err
	}

	t, err := time.ParseDuration(s)
	if err != nil {
		return err
	}

	d.duration = t
	return nil
}

// Nanoseconds returns the Duration in nanoseconds
func (d *Duration) Nanoseconds() uint64 {
	return uint64(d.duration.Nanoseconds())
}

// Milliseconds returns the Duration in milliseconds
func (d *Duration) Milliseconds() uint64 {
	return d.Nanoseconds() / 1000000
}

// Seconds returns the Duration in seconds
func (d *Duration) Seconds() float64 {
	return d.duration.Seconds()
}

// String returns the Duration as string
func (d *Duration) String() string {
	return d.duration.String()
}
