package models

import "time"

// PublishingMode is a Settings.PublishingMode
type PublishingMode string

const (
	// PublishingModeOpen allows all comments and publishes them (if they pass AutoMod [if enabled])
	PublishingModeOpen PublishingMode = "open"
	// PublishingModeReview holds all new comments for review
	PublishingModeReview PublishingMode = "review"
	// PublishingModeDisabled does not allow any comments
	PublishingModeDisabled PublishingMode = "disabled"
)

var (
	validPublishingModes = map[PublishingMode]struct{}{PublishingModeOpen: {}, PublishingModeReview: {}, PublishingModeDisabled: {}}
)

// Settings holds all flipper settings for a channel
type Settings struct {
	ChannelID             string          `json:"channel_id"`
	CreatedAt             *time.Time      `json:"created_at"`
	UpdatedAt             *time.Time      `json:"updated_at"`
	PublishingMode        *PublishingMode `json:"publishing_mode"`
	FollowersOnlyDuration *Duration       `json:"followers_only_duration"`
}

// SettingsUpdate holds setting updates
type SettingsUpdate struct {
	PublishingMode        *string    `json:"publishing_mode"`
	UpdatedAt             *time.Time `json:"updated_at"`
	FollowersOnlyDuration *Duration  `json:"followers_only_duration"`
}

// IsValidPublishingMode returns true if publishingMode is a valid PublishingMode
func IsValidPublishingMode(publishingMode string) bool {
	_, ok := validPublishingModes[PublishingMode(publishingMode)]
	return ok
}
