package models

import "time"

const (
	// VODBroadcastTypeArchive is the BroadcastType of past broadcasts
	VODBroadcastTypeArchive = "archive"
	// VODBroadcastTypePastPremiere is the BroadcastType of past premieres
	VODBroadcastTypePastPremiere = "past_premiere"
	// VODBroadcastTypeUpload is the BroadcastType of uploads
	VODBroadcastTypeUpload = "upload"
	// VODBroadcastTypeHighlight is the BroadcastType of highlights
	VODBroadcastTypeHighlight = "highlight"
)

// VOD represents a Twitch VOD
type VOD struct {
	ID                string
	BroadcastType     string
	BroadcastID       string
	HandwaveLatencyMS *int64
	OwnerID           string
	Duration          *ContentOffset
	StartedOn         *time.Time
	Viewable          *string
	ViewableAt        *time.Time
	Offset            *ContentOffset
}

// CalculateDuration returns the VOD's duration
func (v *VOD) CalculateDuration() *ContentOffset {
	if v.IsRecording() {
		return NewContentOffset(time.Since(*v.StartedOn))
	}
	return NewContentOffset(time.Duration(v.Duration.Nanoseconds()))
}

// IsPublic returns true if the VOD is public
func (v *VOD) IsPublic() bool {
	if v.Viewable != nil && *v.Viewable == "public" {
		return true
	}
	if v.ViewableAt != nil && v.ViewableAt.Before(time.Now()) {
		return true
	}
	return false
}

// IsRecording returns true if the VOD is still being recorded
func (v *VOD) IsRecording() bool {
	return v.Duration == nil
}
