package jax

import (
	"encoding/json"

	"code.justin.tv/web/jax/common/usher"
)

// Streams represents the results returned by Jax
type Streams struct {
	TotalCount int       `json:"_total"`
	Hits       []*Stream `json:"hits"`
}

// Stream represents a single Jax stream
type Stream struct {
	Channel    string           `json:"channel"`
	Properties StreamProperties `json:"properties"`
}

// StreamProperties are usually associated with a Jax stream
type StreamProperties struct {
	Rails         *StreamRailsProperties         `json:"rails"`
	Zuma          *StreamZumaProperties          `json:"zuma"`
	Usher         *StreamUsherProperties         `json:"usher"`
	CSGO          *StreamCSGOProperties          `json:"csgo"`
	LoL           *StreamLoLProperties           `json:"leagueoflegends"`
	PlayStation   *StreamPlayStationProperties   `json:"playstation"`
	XboxHeartbeat *StreamXboxHeartbeatProperties `json:"xbox_heartbeat"`
	FilterData    *StreamFilterProperties        `json:"filterdata"`
	Hearthstone   *StreamHearthstoneProperties   `json:"hearthstone"`
	PUBG          *StreamPUBGProperties          `json:"pubg"`
	Overwatch     *StreamOverwatchProperties     `json:"overwatch"`
	Partnerships  *StreamPartnershipsProperties  `json:"partnerships"`
}

// StreamCSGOProperties are only present for CS:GO streams with an
// associated Steam account
type StreamCSGOProperties struct {
	LastUpdated *string                `json:"last_updated"`
	Map         *string                `json:"map"`
	MapImg      *string                `json:"map_img"`
	MapName     *string                `json:"map_name"`
	Skill       *int                   `json:"skill"`
	Spectators  *int                   `json:"spectators"`
	TeamCT      map[string]interface{} `json:"teamCT"`
	TeamT       map[string]interface{} `json:"teamT"`
}

// StreamLoLProperties are only present for League of Legends streams with an
// associated League of Legends account
type StreamLoLProperties struct {
	SummonerName     *string `json:"summoner_name"`
	SummonerRank     *string `json:"summoner_rank"`
	SummonerDivision *string `json:"summoner_division"`
	SummonerID       *int    `json:"summoner_id"`
	ChampionID       *int    `json:"champion_id"`
}

// StreamHearthstoneProperties are only present for Hearthstone streams with deepmetadata available
type StreamHearthstoneProperties struct {
	BroadcasterHeroName  string `json:"broadcaster_hero_name,omitempty"`
	BroadcasterHeroClass string `json:"broadcaster_hero_class,omitempty"`
	OpponentHeroName     string `json:"opponent_hero_name,omitempty"`
	OpponentHeroClass    string `json:"opponent_hero_class,omitempty"`
	HSGameMode           string `json:"game_mode,omitempty"`
}

// StreamPUBGProperties are only present for PUBG streams with deepmetadata available
type StreamPUBGProperties struct {
	PlayerAliveCount int    `json:"player_alive_count,omitempty"`
	PUBGGameMode     string `json:"game_mode,omitempty"`
}

// StreamOverwatchProperties are only present for Overwatch streams with deepmetadata available
type StreamOverwatchProperties struct {
	BroadcasterCharacter        string `json:"broadcaster_character,omitempty"`
	BroadcasterCharacterRole    string `json:"broadcaster_character_role,omitempty"`
	BroadcasterCharacterAbility string `json:"broadcaster_character_ability,omitempty"`
}

// FilterProperties are only present for HS and Overwatch streams (for now)
type StreamFilterProperties struct {
	FilterData *string `json:"filterdata"`
}

// StreamXboxHeartbeatProblems contains additional information for users streaming though
// the Twitch Xbox app and users streaming Xbox games through a capture card (with an Xbox Live
// ID connected to their Twitch account)
type StreamXboxHeartbeatProperties struct {
	LastUpdated *string `json:"last_updated"`
	Live        *bool   `json:"live"`
}

// StreamPlayStationProperties contains additional information for
// PlayStation streams
type StreamPlayStationProperties struct {
	LastUpdated                   *string `json:"last_updated"`
	SCEPlatform                   *string `json:"sce_platform"`
	SCETitleAgeRating             *int    `json:"sce_title_age_rating"`
	SCETitleAttribute             *string `json:"sce_title_attribute"`
	SCETitleGenre                 *string `json:"sce_title_genre"`
	SCETitleID                    *string `json:"sce_title_id"`
	SCETitleLanguage              *string `json:"sce_title_language"`
	SCETitleMetadata              *string `json:"sce_title_metadata"`
	SCETitleName                  *string `json:"sce_title_name"`
	SCETitlePreset                *bool   `json:"sce_title_preset"`
	SCETitlePresetText1           *string `json:"sce_title_preset_text_1"`
	SCETitlePresetText2           *string `json:"sce_title_preset_text_2"`
	SCETitlePresetText3           *string `json:"sce_title_preset_text_3"`
	SCETitlePresetText4           *string `json:"sce_title_preset_text_4"`
	SCETitlePresetText5           *string `json:"sce_title_preset_text_5"`
	SCETitlePresetTextDescription *string `json:"sce_title_preset_text_description"`
	SCETitleProductID             *string `json:"sce_title_product_id"`
	SCETitleSessionID             *string `json:"sce_title_session_id"`
	SCETitleShortName             *string `json:"sce_title_short_name"`
	SCETitleStoreURL              *string `json:"sce_title_store_url"`
	SCEUserCountry                *string `json:"sce_user_country"`
	SCEUserNPID                   *string `json:"sce_user_np_id"`
	SCEUserOnlineID               *string `json:"sce_user_online_id"`
}

// StreamUsherProperties is what Jax returns as `properties.usher`.
// As the name suggests, this data originally comes from Usher and
// contains video related properties.
type StreamUsherProperties struct {
	AVCLevel          *string  `json:"avc_level"`
	AVCProfile        *string  `json:"avc_profile"`
	AverageFPS        *float64 `json:"average_fps"`
	BroadcastPart     *int     `json:"broadcast_part"`
	Broadcaster       *string  `json:"broadcaster"`
	BroadcastPlatform *string  `json:"broadcast_platform"`
	ChannelCount      *int     `json:"channel_count"`
	ClientID          *string  `json:"client_id"`
	Configuration     *string  `json:"configuration"`
	DelayLength       *int     `json:"delay_length"`
	HLS               *bool    `json:"hls"`
	ID                *int     `json:"id"`
	LastUpdated       *string  `json:"last_updated"`
	MaxHeight         *int     `json:"max_height"`
	Name              *string  `json:"name"`
	StreamType        *string  `json:"stream_type"` // TODO deprecate this in favor of broadcast_platform
	StreamUpTimestamp *int     `json:"stream_up_timestamp"`
	UpTime            *string  `json:"up_time"`
	UpdatedOn         *string  `json:"updated_on"`
	VideoBitrate      *float64 `json:"video_bitrate"`
	VideoCodec        *string  `json:"video_codec"`
	VideoHeight       *int     `json:"video_height"`
	VideoWidth        *int     `json:"video_width"`
}

//A differently named type to avoid infinite recursion when unmarshalling
type streamUsherProperties StreamUsherProperties

//UnmarshalJSON unmarshals the stream usher properties and sets
//any derived fields.
func (s *StreamUsherProperties) UnmarshalJSON(b []byte) error {
	properties := streamUsherProperties{}
	err := json.Unmarshal(b, &properties)

	if err != nil {
		return err
	}

	if properties.Broadcaster != nil {
		broadcastPlatform := usher.BroadcastPlatformMapping(*properties.Broadcaster)
		streamType := usher.StreamTypeMapping(*properties.Broadcaster)

		properties.BroadcastPlatform = &broadcastPlatform
		properties.StreamType = &streamType
	}

	*s = StreamUsherProperties(properties)

	return nil
}

// StreamRailsProperties is what Jax returns as `properties.rails`
// This data originally comes from Rails and contains channel/user
// related properties.
type StreamRailsProperties struct {
	BroadcasterLanguage *string `json:"broadcaster_language"`
	Category            *string `json:"category"`
	Channel             *string `json:"channel"`
	ChannelID           *int    `json:"channel_id"`
	ChannelViewCount    *int    `json:"channel_view_count"`
	DirectoryHidden     *bool   `json:"directory_hidden"`
	GameID              *int    `json:"game_id"`
	Language            *string `json:"language"`
	LastUpdated         *string `json:"last_updated"`
	MetaGame            *string `json:"meta_game"`
	SteamID             *string `json:"steam_id"`
	Title               *string `json:"title"`
	XboxGamerTag        *string `json:"xbox_gamertag"`
	XboxTitleID         *string `json:"xbox_title_id"`
}

// StreamZumaProperties is what Jax returns as `properties.zuma`
// This data originally comes from Zuma and contains community-related
// properties.
type StreamZumaProperties struct {
	CommunityID  *string  `json:"community_id"`
	CommunityIDs []string `json:"community_ids"`
}

type StreamPartnershipsProperties struct {
	PartnerProgram *bool `json:"partner_program"`
}

// StreamSummary is what Jax returns for summary of live streams.
type StreamSummary struct {
	Total   int                   `json:"total"`
	Results []StreamSummaryResult `json:"results"`
}

type StreamSummaryResult struct {
	Channels  int         `json:"channels"`
	Viewers   int         `json:"viewers"`
	SortValue *int        `json:"sort_value"`
	GroupBy   interface{} `json:"group_by,omitempty"`
}

// Needed to be able to unmarshal arbitrary group-by field names
func (s *StreamSummaryResult) UnmarshalJSON(b []byte) error {
	var groupBy map[string]interface{}

	err := json.Unmarshal(b, &groupBy)

	if err != nil {
		return err
	}

	var summary StreamSummaryResult

	if _, ok := groupBy["channels"]; ok {
		if channels, ok := groupBy["channels"].(float64); ok {
			summary.Channels = int(channels)
		}
		// remove from map after processing
		delete(groupBy, "channels")
	}

	if _, ok := groupBy["viewers"]; ok {
		if viewers, ok := groupBy["viewers"].(float64); ok {
			summary.Viewers = int(viewers)
		}
		// remove from map after processing
		delete(groupBy, "viewers")
	}

	if _, ok := groupBy["sort_value"]; ok {
		if sortValue, ok := groupBy["sort_value"].(float64); ok {
			sortPoint := int(sortValue)
			summary.SortValue = &sortPoint
		}
		// remove from map after processing
		delete(groupBy, "sort_value")
	}

	// response of jax always only contains one group by key
	for _, value := range groupBy {
		summary.GroupBy = value
	}

	*s = summary

	return nil
}
