package jax

import (
	"net/url"
	"strconv"
	"strings"
)

const (
	HSGameModeField       = "hearthstone.game_mode"
	HSRankAllValue        = "rank_legendary,rank_01,rank_02,rank_03,rank_04,rank_05,rank_06,rank_07,rank_08,rank_09,rank_10,rank_11,rank_12,rank_13,rank_14,rank_15,rank_16,rank_17,rank_18,rank_19,rank_20,rank_21,rank_22,rank_23,rank_24,rank_25"
	HSArenaAllValue       = "arena_0,arena_1,arena_2,arena_3,arena_4,arena_5,arena_6,arena_7,arena_8,arena_9,arena_10,arena_11"
	HSTavernBrawlAllValue = "tavern_brawl_regular,tavern_brawl_heroic_00,tavern_brawl_heroic_01,tavern_brawl_heroic_02,tavern_brawl_heroic_03,tavern_brawl_heroic_04,tavern_brawl_heroic_05,tavern_brawl_heroic_06,tavern_brawl_heroic_07,tavern_brawl_heroic_08,tavern_brawl_heroic_09,tavern_brawl_heroic_10,tavern_brawl_heroic_11"
)

type StreamOptions struct {
	Fields       []string
	ExistsFields []string
	Filters      []Filter
}

type StreamsOptions struct {
	Pagination   PaginationOptions
	Sort         string
	Fields       []string
	ExistsFields []string
	Filters      []Filter
}

type StreamSummaryOptions struct {
	Pagination       PaginationOptions
	GroupBy          string
	Sort             string
	IncludeSortValue bool
	Filters          []Filter
}

type PaginationOptions struct {
	Limit  int
	Offset int
}

type Filter struct {
	field string
	value string
}

func EqualityFilter(field, value string) Filter {
	return Filter{
		field: field,
		value: value,
	}
}

func MaxHeightFilter(maxHeight string) Filter {
	return Filter{
		field: "usher.max_height",
		value: maxHeight,
	}
}

func XboxHeartbeatFilter(live bool) Filter {
	return Filter{
		field: "xbox_heartbeat",
		value: strconv.FormatBool(live),
	}
}

func HSGameModeFilterRankAll() Filter {
	return Filter{
		field: "hearthstone.game_mode",
		value: HSRankAllValue,
	}
}

func HSGameModeFilterArenaAll() Filter {
	return Filter{
		field: "hearthstone.game_mode",
		value: HSArenaAllValue,
	}
}

func HSGameModeFilterTavernBrawlAll() Filter {
	return Filter{
		field: "hearthstone.game_mode",
		value: HSTavernBrawlAllValue,
	}
}

func StreamTypeFilter(streamType string) Filter {
	return Filter{
		field: "stream_type",
		value: streamType,
	}
}

func (R *StreamOptions) urlParams() url.Values {
	u := url.Values{}
	if R == nil {
		return u
	}

	if len(R.Fields) > 0 {
		u.Set("fields", strings.Join(R.Fields, ","))
	}

	if len(R.ExistsFields) > 0 {
		u.Set("exists_fields", strings.Join(R.ExistsFields, ","))
	}

	for _, f := range R.Filters {
		u.Set(f.field, f.value)
	}
	return u
}

func (R *StreamsOptions) urlParams() url.Values {
	u := url.Values{}
	if R == nil {
		return u
	}

	if R.Pagination.Limit > 0 {
		u.Set("size", strconv.Itoa(R.Pagination.Limit))
	}
	if R.Pagination.Offset > 0 {
		u.Set("from", strconv.Itoa(R.Pagination.Offset))
	}
	if len(R.Fields) > 0 {
		u.Set("fields", strings.Join(R.Fields, ","))
	}
	if len(R.ExistsFields) > 0 {
		u.Set("exists_fields", strings.Join(R.ExistsFields, ","))
	}
	if len(R.Sort) > 0 {
		u.Set("sort", R.Sort)
	}

	for _, f := range R.Filters {
		u.Set(f.field, f.value)
	}
	return u
}

func (R *StreamSummaryOptions) urlParams() url.Values {
	u := url.Values{}
	if R == nil {
		return u
	}

	if len(R.GroupBy) > 0 {
		u.Set("group-by", R.GroupBy)
	}

	if len(R.Sort) > 0 {
		u.Set("sort", R.Sort)
	}

	for _, f := range R.Filters {
		u.Set(f.field, f.value)
	}

	return u
}
