// Common plugins
const webpack = require("webpack");
const HtmlWebPackPlugin = require("html-webpack-plugin");
const ForkTsCheckerWebpackPlugin = require("fork-ts-checker-webpack-plugin");
const MiniCssExtractPlugin = require("mini-css-extract-plugin");

// Minify plugins
const TerserPlugin = require("terser-webpack-plugin");

// Dev plugins
const { prepareProxy } = require("react-dev-utils/WebpackDevServerUtils");

// Paths and proxies
const path = require("path");
const paths = require("./config/paths");
const proxySetting = require(paths.appPackageJson).proxy;
const proxy = prepareProxy(proxySetting, paths.appPublic);

const publicPath = "/";

module.exports = (env, argv) => {
  // Environment variables
  const productionMode = argv.mode === "production";
  process.env.BABEL_ENV = argv.mode;
  process.env.NODE_ENV = argv.mode;
  require("./config/env");

  // Options that vary based on mode are defined outside of the return
  let optimization = {};
  let performance = {};
  let plugins = [];
  let output = {
    path: paths.appBuild,
    pathinfo: true,
    filename: "static/js/bundle.js",
    // There are also additional JS chunk files if you use code splitting.
    chunkFilename: "static/js/[name].chunk.js",
    // This is the URL that app is served from. We use "/" in development.
    publicPath: publicPath
  };

  // Config for output in cli
  let stats = {
    assets: true,
    children: true,
    entrypoints: false,
    chunkModules: false,
    chunks: false,
    colors: true,
    hash: false,
    timings: false,
    version: false,
    modules: false,
    warningsFilter: /export .* was not found in/
  };

  // Production mode/build options
  if (productionMode) {
    // Configuration for Webpack 4 built-in optimizer
    optimization = {
      // Minimize with UglifyJS + OptimizeCSS
      minimizer: [
        new TerserPlugin({
          parallel: true,
          sourceMap: true,
          terserOptions: {
            ecma: 6,
            compress: {
              ecma: 6
            }
          }
        })
      ],
      // Providing this field turns off all default Webpack 4 minimization
      splitChunks: {
        cacheGroups: {
          vendor: {
            test: /node_modules/,
            name: "vendor",
            chunks: "initial",
            enforce: true
          }
        }
      }
    };
    // Max size before warnings are printed
    performance = {
      maxEntrypointSize: 1000000,
      maxAssetSize: 1000000
    };
    output = {
      ...output,
      filename: "static/js/bundle.[hash].js",
      chunkFilename: "static/js/[name].[chunkhash].js"
    };
    plugins = [
      ...plugins,
      new MiniCssExtractPlugin({
        filename: "[name].[chunkhash].css"
      })
    ];
  } else {
    // Increased max size further as dev is not bundled or minified
    performance = {
      hints: false,
      maxEntrypointSize: 10000000,
      maxAssetSize: 10000000
    };
    // Additional noise suppression for development
    stats = {
      ...stats,
      assets: false,
      builtAt: false
    };
  }

  return {
    context: __dirname,
    entry: paths.appInit,
    module: {
      rules: [
        {
          test: /\.(ts|tsx)$/,
          loader: require.resolve("tslint-loader"),
          enforce: "pre",
          include: paths.appSrc
        },
        {
          test: /\.js$/,
          loader: require.resolve("source-map-loader"),
          enforce: "pre",
          include: paths.appSrc
        },
        {
          test: /\.(ts|tsx)$/,
          use: [
            require.resolve("cache-loader"),
            require.resolve("thread-loader"), // Parallelizes all following loaders
            require.resolve("babel-loader")
          ]
        },
        {
          test: /\.(scss|sass|css)$/,
          use: [
            productionMode ? MiniCssExtractPlugin.loader : require.resolve("style-loader"),
            require.resolve("css-loader"),
            {
              loader: 'postcss-loader',
              options: {
                plugins: [
                  require('autoprefixer')(),
                  require('css-mqpacker')({ sort: true }),
                ],
              },
            },
            {
              loader: require.resolve("sass-loader"), // compiles Sass to CSS
              options: {
                includePaths: [path.resolve(paths.appNodeModules, "twitch-core-ui/")]
              }
            }
          ]
        }
      ]
    },
    resolve: {
      extensions: [".ts", ".tsx", ".js", ".json", ".jsx"],
      alias: {
        aegis: path.join(__dirname, "./src")
      }
    },
    devtool: productionMode ? "source-map" : "inline-source-map",
    devServer: {
      clientLogLevel: "error",
      contentBase: paths.appPublic,
      compress: true,
      headers: {
        "X-Frame-Options": "deny"
      },
      host: process.env.HOST || "0.0.0.0",
      inline: true,
      watchOptions: {
        ignored: /node_modules/
      },
      stats, // webpack-dev-server needs stats specified separate from config
      proxy,
      overlay: true,
      historyApiFallback: true,
      port: 3000
    },
    output,
    optimization,
    performance,
    stats,
    plugins: [
      ...plugins,
      /*
            ts-loader prints lots of bad errors when compiled in
            "HappyPackMode" (parallel). Checking types separately
            with ForkTsChecker allows for compilation errors to
            be "ignored" but still visible. See:
            https://github.com/TypeStrong/ts-loader#happypackmode-boolean-defaultfalse
            */
      new ForkTsCheckerWebpackPlugin({
        checkSyntacticErrors: true,
        workers: 2
      }),
      new HtmlWebPackPlugin({
        favicon: "public/favicon.ico",
        inject: true,
        template: paths.appHtml,
        minify: {
          removeComments: true,
          collapseWhitespace: true,
          removeRedundantAttributes: true,
          useShortDoctype: true,
          removeEmptyAttributes: true,
          removeStyleLinkTypeAttributes: true,
          keepClosingSlash: true,
          minifyJS: true,
          minifyCSS: true,
          minifyURLs: true
        }
      }),
      new webpack.IgnorePlugin(/^\.\/locale$/, /moment$/)
    ]
  };
};
