import socket
import subprocess
import boto3

from etcd_api import EtcD
from config import cfg
from ec2 import EC2

from starkutil import es_event

bebo_region = cfg.REGION
ec2 = EC2(bebo_region)

from logger import log

def generate_change_resource_record_set(hostname, identifier, ip, ttl=cfg.ROUTE53_TTL):
    return {
        'Type': 'A',
        'Name': hostname,
        'SetIdentifier': identifier,
        'MultiValueAnswer': True,
        'TTL': ttl,
        'ResourceRecords': [{
            'Value': ip
        }]
    }

def generate_change_alias(from_hostname, identifier, hosted_zone_id, to_hostname, region=ec2.region, ttl=cfg.ROUTE53_TTL):
    return {
        'Name': from_hostname,
        'Type': 'A',
        'Region': region,
        'SetIdentifier': identifier,
        'AliasTarget': {
            'HostedZoneId': hosted_zone_id,
            'DNSName': to_hostname,
            'EvaluateTargetHealth': False
        }
    }


class Route53Controller(object):
    etcd_api = EtcD()
    bebo_domain = cfg.BEBO_DOMAIN
    internal_domain = cfg.INTERNAL_DOMAIN
    route53_timeout = cfg.ROUTE53_TIMEOUT
    cached_external_hosted_zone_id = None
    cached_internal_hosted_zone_id = None
    """
    Adds and removed Route53 DNS records
    """
    def __init__(self, cluster_name, logic):
        self.cluster_name = cluster_name
        self.logic = logic
        self.etcd_api = Route53Controller.etcd_api

        self.global_record_name = '{}.{}.'.format(self.cluster_name, self.bebo_domain)
        self.regional_public_record_name = '{}-{}.{}.'.format(bebo_region, self.cluster_name, self.bebo_domain)
        self.regional_private_record_name = '{}-{}.{}.'.format(bebo_region, self.cluster_name, self.internal_domain)

        self.create_boto_session()
        self.create_route53_client()

    def create_boto_session(self):
        self.session = boto3.session.Session()

    def create_route53_client(self):
        self.client = self.session.client('route53', region_name=ec2.region)

    def external_hosted_zone_id(self):
        if not self.cached_external_hosted_zone_id:
            try:
                self.cached_external_hosted_zone_id = self.client.list_hosted_zones_by_name(
                    DNSName=self.bebo_domain,
                    MaxItems='1'
                ).get('HostedZones', list())[0].get('Id')
            except (IndexError, KeyError):
                self.cached_external_hosted_zone_id = None

        return self.cached_external_hosted_zone_id

    def internal_hosted_zone_id(self):
        if not self.cached_internal_hosted_zone_id:
            try:
                self.cached_internal_hosted_zone_id = self.client.list_hosted_zones_by_name(
                    DNSName=self.internal_domain,
                    MaxItems='1'
                ).get('HostedZones', list())[0].get('Id')
            except (IndexError, KeyError):
                self.cached_internal_hosted_zone_id = None

        return self.cached_internal_hosted_zone_id


    def _change_resource_record_sets(self, lock_key, changes, comment, hosted_zone_id):
        if cfg.DRY_RUN:
            log.info("[{}] DRY RUN _change_resource_record_sets on {}".format(self.cluster_name, comment))
            return

        if self.etcd_api.get(lock_key):
            log.debug('[{}] Not acting on {} because lock exists'.format(self.cluster_name, comment))
            return

        log.info('[{}] Acting on "{}"'.format(self.cluster_name, comment))

        change_batch = {
            'Comment': comment,
            'Changes': changes
            }
        try:
            self.client.change_resource_record_sets(
                HostedZoneId=hosted_zone_id,
                ChangeBatch=change_batch
            )
            self.etcd_api.set(lock_key, True, ttl=self.route53_timeout)
        except Exception as e:
            log.exception('[{}] Failed to act on {} because of ({})'.format(self.cluster_name, comment, changes))

    @staticmethod
    def get_ips_for_record(record_name):
        if not record_name:
            return []
        command = ['host', record_name]
        process = subprocess.Popen(command, stdout=subprocess.PIPE)
        out, err = process.communicate()
        if err is not None:
            log.error('ERROR resolve_dns_record: {}'.format(err))
            return []

        out = out.decode('utf-8')
        if 'NXDOMAIN' in out:
            return []

        return list({line.split('has address ')[1] for line in [line for line in out.split("\n") if "has address" in line]})


    def upsert_public_record(self, box):
        public_ip = box.public_ip
        public_hostname = box.public_hostname
        if not public_ip:
            log.warn('[{}] Cannot upsert public record -- no public ip for {}'.format(self.cluster_name, box))
            return

        if not public_hostname:
            log.warn('[{}] Cannot upsert public record -- no public hostname for {}'.format(self.cluster_name, box))
            return

        lock_key = 'route53/{}'.format(public_hostname)

        changes = [{
            'Action': 'UPSERT',
            'ResourceRecordSet': generate_change_resource_record_set(public_hostname, box.instance_id, public_ip)
        }]

        comment = 'Creating public DNS record for {} --> {}'.format(public_hostname, public_ip)

        self._change_resource_record_sets(lock_key, changes, comment, self.external_hosted_zone_id())

    def destroy_public_record(self, box):
        public_ip = box.public_ip
        resolved_ip = box.resolved_ip
        public_hostname = box.public_hostname

        if not resolved_ip:
            log.warn('[{}] Cannot destroy public record -- no resolved IP for {}'.format(self.cluster_name, box))
            return

        if not public_ip:
            log.warn('[{}] Cannot destroy public record -- no public ip for {}'.format(self.cluster_name, box))
            return

        if not public_hostname:
            log.warn('[{}] Cannot destroy public record -- no public hostname for {}'.format(self.cluster_name, box))
            return

        lock_key = 'route53/{}'.format(public_hostname)

        changes = [{
            'Action': 'DELETE',
            'ResourceRecordSet': generate_change_resource_record_set(public_hostname, box.instance_id, resolved_ip)
        }]
        comment = 'Destroying public DNS record for {} --> {}'.format(public_hostname, resolved_ip)

        self._change_resource_record_sets(lock_key, changes, comment, self.external_hosted_zone_id())

    def ensure_global_endpoint(self):
        log.info('[{}] ensure_global_endpoint'.format(self.cluster_name))
        lock_key = 'route53/{}'.format(self.global_record_name)

        regional_endpoint_healthy = bool(self.logic.get_healthy())
        hosted_zone_id = self.external_hosted_zone_id().split("/")[-1]

        global_present_ips = sorted(list(set(self.get_ips_for_record(self.global_record_name))))
        regional_present_ips = sorted(list(set(self.get_ips_for_record(self.regional_public_record_name))))

        regional_exists_in_global = False
        for ip in global_present_ips:
            if ip in regional_present_ips:
                regional_exists_in_global = True
                break

        action = None
        comment = ""

        if not regional_endpoint_healthy and regional_exists_in_global:
            action = "DELETE"
            comment = 'Remove {} from global {} cluster'.format(bebo_region, self.cluster_name)
        elif regional_endpoint_healthy and not regional_exists_in_global:
            action = "CREATE"
            comment = 'Add {} to global {} cluster'.format(bebo_region, self.cluster_name)

        data_dict = {'regional_endpoint_healthy': regional_endpoint_healthy, 'regional_exists_in_global': regional_exists_in_global, 'action': action}

        es_event('ensure_dns', 'global', label_tx=self.cluster_name, data=data_dict)

        if action:
            changes = [{
                'Action': action,
                'ResourceRecordSet': generate_change_alias(self.global_record_name, bebo_region, hosted_zone_id, self.regional_public_record_name)
            }]

            self._change_resource_record_sets(lock_key, changes, comment, self.external_hosted_zone_id())

    def ensure_public_regional_endpoint(self):
        log.info('[{}] ensure_public_regional_endpoint'.format(self.cluster_name))
        healthy_public_ips = [host.public_ip for host in self.logic.get_healthy() if host.public_ip is not None]
        unhealthy_public_ips = [host.public_ip for host in self.logic.get_cluster_boxes() if host.public_ip is not None and host.public_ip not in healthy_public_ips] #either unhealthy or cant take more load

        es_event('ensure_dns', 'public_regional', label_tx=self.cluster_name, data={'healthy_public_ips_nr': len(healthy_public_ips), 'unhealthy_public_ips_nr': len(unhealthy_public_ips)})

        self._ensure_a_record_endpoint(self.regional_public_record_name, healthy_public_ips, unhealthy_public_ips, self.external_hosted_zone_id())

    def ensure_private_regional_endpoint(self):
        healthy_private_ips = [host.private_ip for host in self.logic.get_healthy() if host.private_ip is not None]
        unhealthy_private_ips = [host.private_ip for host in self.logic.get_cluster_boxes() if host.public_ip is not None and host.private_ip not in healthy_private_ips] #either unhealthy or cant take more load

        es_event('ensure_dns','private_regional', label_tx=self.cluster_name, data={'healthy_private_ips_nr': len(healthy_private_ips), 'unhealthy_private_ips_nr': len(unhealthy_private_ips)})

        self._ensure_a_record_endpoint(self.regional_private_record_name, healthy_private_ips, unhealthy_private_ips, self.internal_hosted_zone_id())

    def _ensure_a_record_endpoint(self, hostname, healthy_ips, unhealthy_ips, hosted_zone_id):
        lock_key = 'route53/{}'.format(hostname)

        present_ips = self.get_ips_for_record(hostname)

        log.info('[{}] hostname: {}, healthy_ips: {}, unhealthy_ips: {}, present_ips: {}'.format(self.cluster_name, hostname, healthy_ips, unhealthy_ips, present_ips))

        changes = []
        to_add = [ip for ip in healthy_ips if ip not in present_ips]
        for ip in to_add:
            change = {
                'Action': 'CREATE',
                'ResourceRecordSet': generate_change_resource_record_set(hostname, ip, ip)
                }
            changes.append(change)

        to_remove = [ip for ip in present_ips if ip not in healthy_ips]
        for ip in to_remove:
            change = {
                'Action': 'DELETE',
                'ResourceRecordSet': generate_change_resource_record_set(hostname, ip, ip)
                }
            changes.append(change)

        if changes:
            comment = 'Remove {} and add {} {} boxes to {}'.format(len(to_remove), len(to_add), self.cluster_name, hostname)

            es_event('ensure_dns', 'update_regional', label_tx=self.cluster_name, data={'hostname_tx': hostname, 'add_nr': len(to_add), 'remove_nr': len(to_remove), 'comment_tx': comment})

            self._change_resource_record_sets(lock_key, changes, comment, hosted_zone_id)
