import os
import sys
import time
import traceback

import sentry_sdk
from bebo.utils import init_sentry
from starkutil import timeout, es_event, slack_notifications
from etcd_api import EtcD
from config import cfg
from ec2 import EC2
from logic import StarkLogic
from controllers import ServerController
from salt_api import SaltApi
from models import ClusterModel
from route53 import Route53Controller

from logger import log

init_sentry(cfg.SENTRY_URL)


class Stark(object):
    """
    Make high-level decisions about when to scale, create, and stop boxes.
    """
    def __init__(self):
        self.logic = {}

        self.etcd = EtcD()

        self.bebo_region = cfg.REGION
        self.ec2 = EC2(self.bebo_region)

        self.clusters = self.get_clusters()
        self.salt_api = SaltApi(self.clusters, self.ec2)
        self.server_controller = ServerController(self.clusters, self.ec2, self.etcd)

    def get_clusters(self):
        return [cluster for cluster in ClusterModel.get_all() if not cluster.ignore]

    def update_state(self):
        log.info("update_state")
        self.clusters = self.get_clusters()
        self.server_controller.update_clusters(self.clusters)
        self.salt_api.update_clusters(self.clusters)
        self.server_controller.update_state()

    def scale(self, cluster):
        """ figure out how many boxes we need to create, start, stop, destroy
            store accurate state / action in db """

        cluster_name = cluster.name
        log.info("[{}] scale".format(cluster_name))
        if cluster_name not in self.logic:
            self.logic[cluster_name] = StarkLogic(self.ec2, self.salt_api)

        self.logic[cluster_name].scale(cluster)

    def stark(self):
        log.info("running stark main loop")
        self.update_state()
        log.info("self.clusters: {}".format([cluster.name for cluster in self.clusters]))
        es_event('clusters', 'list', data={'cluster_num_nr': len(self.clusters), 'clusters_names': [cluster.name for cluster in self.clusters] })
        for cluster in self.clusters:
            try:
                self.scale(cluster)
            except Exception as e:
                log.exception("Failed to scale {}".format(cluster.name))

    def run(self):
        while True:
            log.info("starting stark run loop")
            with timeout(seconds=60):
                try:
                    self.stark()
                except:
                    log.exception("stark run loop generic error")
                time.sleep(cfg.STARK_INTERVAL)

def main():
    log.info('Auto-scaler starting')
    try:
        stark = Stark()
        log.info("stark init")
        stark.run()
    except KeyboardInterrupt:
        raise KeyboardInterrupt
    except Exception:
        log.exception("catch all excpetion in main loop")
        time.sleep(cfg.STARK_INTERVAL)

        exc_type, _, exc_tb = sys.exc_info()
        fn = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
        err = '{}. File: {}, line {}. Full: {}'
        stack_trace = err.format(exc_type, fn, exc_tb.tb_lineno, traceback.format_exc())
        es_event('error', "main_loop", data={'stack_trace_tx': stack_trace})
        slack_notifications(stack_trace)
        main()

if __name__ == '__main__':
    main()
