import json
import os
import signal
import sys
import traceback
from enum import Enum
from builtins import TimeoutError

import requests

from bebo import dispatcher, utils

from config import cfg

from logger import log

region_request = requests.get('http://169.254.169.254/latest/meta-data/placement/availability-zone', timeout=0.5)
aws_region = region_request.text[:-1]

class timeout:
    def __init__(self, seconds=100, error_message='Timeout'):
        self.seconds = seconds
        self.error_message = error_message

    def handle_timeout(self, *_):
        raise TimeoutError(self.error_message)

    def __enter__(self):
        signal.signal(signal.SIGALRM, self.handle_timeout)
        signal.alarm(self.seconds)

    def __exit__(self, _, value, __):
        signal.alarm(0)


def es_event(category_tx, action_tx, label_tx='', data=None):
    if not data:
        data = {}
    queue_name = 'stark_es'

    data['region_tx'] = cfg.REGION
    data['category_tx'] = category_tx# bebo_region
    data['action_tx'] = action_tx # cluster
    data['label_tx'] = label_tx # event we care about
    data['routing_key'] = '{}.{}.{}'.format(data['category_tx'], data['action_tx'], data['label_tx'])

    data['aws_region_tx'] = aws_region
    data['env_tx'] = cfg.BEBO_ENV

    es_data = utils.convert_to_es(data)
    dispatcher.write(queue_name, es_data)

def slack_notifications(msg):
    log.info('Sending Slack Notification: {}'.format(msg))
    if not cfg.SLACK_URL:
        return
    try:
        payload = {'text': '{} \n```{}```'.format(cfg.REGION, msg)}
        resp = requests.post(cfg.SLACK_URL, data=json.dumps(payload), timeout=30)
        if resp.status_code != 200:
            es_event('error', 'slack', data={
                'status_code_nr': resp.status_code,
                'text_tx': resp.text,
                'payload_tx': payload
            })
    except Exception:
        log.exception('Failed to send to slack: {}'.format(msg))
        exc_type, _, exc_tb = sys.exc_info()
        fn = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
        err = '{}. File: {}, line {}. Full: {}'
        stack_trace = err.format(exc_type, fn, exc_tb.tb_lineno, traceback.format_exc())
        log.exception(stack_trace)
        es_event('error', 'slack_except', {'stack_trace_tx': stack_trace})

def health_status_changed(hostname, health_status, new_health_status):
    changed = False
    for k in new_health_status:
        if k not in health_status:
            changed = True
            log.debug('[{}] {} not in health_status -- adding'.format(hostname, k))
        if health_status.get(k, new_health_status[k]) != new_health_status[k]:
            changed = True
            log.debug('[{}] {} value has changed from {} to {}'.format(hostname, k, health_status[k], new_health_status[k]))

    return changed

class StarkStates(Enum):
    CREATING = 'CREATING'
    STARTING = 'STARTING'
    STOPPING = 'STOPPING'
    TERMINATING = 'TERMINATING'
    ERROR = 'ERROR'
    NONE = None
