import React, { StrictMode, memo, Suspense, useEffect, lazy } from 'react';
import { Router, Redirect, Match } from '@reach/router';

import { TunnelProvider, TunnelPlaceholder } from 'react-tunnels';

import { isDesktopApp } from 'scripts/utils';

import { isLoggedIn$ } from 'controllers/user';

//hooks
import useObservable from 'hooks/useObservable';

//end hooks

//components
import TitleBar from 'components/title-bar';
import ErrorBoundary from 'components/ErrorBoundary';
import SnackProvider from 'components/snackbar';
import { FullSizeLoader } from 'bebo-ui';

import AsyncFTUX from 'routes/ftux';
import AsyncDownload from 'routes/download';
import Close from 'routes/close';
import Overlay from 'routes/overlay';
//end components

//routes (deprecated)
const Signup = lazy(() => import('routes/signup'));
const Home = lazy(() => import('routes/home'));
const Success = lazy(() => import('routes/success'));
const Legal = lazy(() => import('routes/legal'));
const NotFound = lazy(() => import('routes/not-found'));
const Invite = lazy(() => import('routes/invite'));
const HowTo = lazy(() => import('routes/how-to'));
const Video = lazy(() => import('routes/video'));

const Join = lazy(() => import('routes/join'));
const MatchRedirect = lazy(() => import('routes/match-redirect'));
const TournamentRedirect = lazy(() => import('routes/tournament-redirect'));
const NotificationRedirect = lazy(() => import('routes/notification'));
//end routes

//pages (new)
const AppPage = lazy(() => import('pages/app'));

//end pages

const AuthRoutes = memo(({ children, uri, ...rest }) => {
  const isLoggedIn = useObservable(isLoggedIn$);
  if (!isLoggedIn) {
    return (
      <Match path="/app/*">
        {({ match }) => {
          if (match) {
            return <Redirect noThrow to={`/login`} />;
          }
          return <Redirect noThrow to={`/login/to${uri}`} />;
        }}
      </Match>
    );
  }
  return children;
});

const App = memo(() => {
  return (
    <StrictMode>
      <ErrorBoundary label="root" notify={true}>
        <SnackProvider>
          <TunnelProvider>
            <>
              {isDesktopApp() && window.location.toString().indexOf('/overlay') === -1 && (
                <TitleBar />
              )}
              <Suspense maxDuration={1000} fallback={<FullSizeLoader />}>
                <Router
                  style={{
                    position: 'absolute',
                    top:
                      isDesktopApp() && window.location.toString().indexOf('/overlay') === -1
                        ? 24
                        : 0,
                    left: 0,
                    right: 0,
                    bottom: 0,
                    display: 'flex'
                  }}
                >
                  {!isDesktopApp() && <Home path="/" default/>}
                  <Legal path="legal/*" />
                  <HowTo path="how-to/*" />
                  {/* this is just for the transition to the new app route */}
                  {/* end this is just for the transition to the new app route */}
                </Router>
              </Suspense>
              <TunnelPlaceholder id={'modal-sheet'} multiple>
                {({ items }) => items.map(({ children }) => children)}
              </TunnelPlaceholder>
            </>
          </TunnelProvider>
        </SnackProvider>
      </ErrorBoundary>
    </StrictMode>
  );
});

export default App;
