import React, { Component } from 'react';
import PropTypes from 'prop-types';

import cx from 'classnames';
import css from './Button.module.scss';

import { Loader } from 'bebo-ui';
import { Link } from '@reach/router';

class Button extends Component {
  static propTypes = {
    children: PropTypes.node,
    color: PropTypes.string,
    size: PropTypes.oneOfType([
      PropTypes.oneOf(['tiny', 'small', 'normal', 'large', 'xlarge']),
      PropTypes.number
    ]),
    onClick: PropTypes.func,
    className: PropTypes.string,
    title: PropTypes.string,
    inverse: PropTypes.bool
  };

  static defaultProps = {
    size: 'normal',
    onClick: event => {}
  };

  static sizes = {
    tiny: '10px',
    small: '12px',
    normal: '14px',
    large: '16px',
    xlarge: '18px'
  };

  generateColor = (primary, inverse) => {
    const colors = ['#000', '#fff'];
    if (inverse) {
      colors.reverse();
    }
    return primary ? colors[0] : colors[1];
  };

  generateChildren() {
    const { loading, disabledText, primary, inverse, children } = this.props;

    if (loading) {
      return (
        <Loader
          className={css.buttonLoader}
          size={16}
          color={this.generateColor(primary, inverse)}
        />
      );
    }
    if (disabledText) {
      return disabledText;
    }
    return children;
  }

  render() {
    const {
      color,
      size,
      onClick,
      compact,
      primary,
      secondary,
      className,
      fullWidth,
      loading,
      disabled,
      active,
      theme,
      link,
      title,
      fluid,
      danger,
      inverse,
      external,
      download,
      style = {}
    } = this.props;

    const styles = Object.assign(
      {},
      {
        color,
        fontSize: Button.sizes[size] || size
      },
      style
    );

    const classes = cx(
      css.button,
      className,
      theme,
      size,
      { [css.primary]: primary },
      { [css.secondary]: secondary },
      { [css.active]: active },
      { [css.fluid]: fluid },
      { [css.danger]: danger },
      { [css.inverse]: inverse },
      { [css.compact]: compact },
      { [css.isDisabled]: disabled },
      { [css.fullWidth]: fullWidth }
    );

    if (link && external) {
      return (
        <a
          target="_blank"
          rel="noopener noreferrer"
          download={download}
          disabled={disabled}
          href={!disabled && link}
          className={classes}
          style={styles}
          onClick={!disabled && !loading ? onClick : null}
        >
          {this.generateChildren()}
        </a>
      );
    } else if (link) {
      return (
        <Link
          disabled={disabled}
          to={(!disabled && link) || ''}
          className={classes}
          style={styles}
          onClick={!disabled && !loading ? onClick : null}
        >
          {this.generateChildren()}
        </Link>
      );
    } else {
      return (
        <button
          disabled={disabled}
          className={classes}
          style={styles}
          onClick={!disabled && !loading ? onClick : null}
          title={title}
        >
          {this.generateChildren()}
        </button>
      );
    }
  }
}

class ButtonGroup extends Component {
  render() {
    const { fluid } = this.props;
    const classes = cx(css.buttonGroup, { [css.fluid]: fluid });
    return <div className={classes}>{this.props.children}</div>;
  }
}

Button.Group = ButtonGroup;

export default Button;
