import React, { memo, useEffect, useRef, useState } from 'react';

import debounce from 'lodash.debounce';
import TextareaAutosize from 'react-textarea-autosize';

import { Icon, Loader } from 'bebo-ui';

import { hasErrorShape, transformErrorShape } from 'scripts/utils';

import cx from 'classnames';
import css from './Input.module.scss';

const debounced = debounce((val, { setSaving, setSuccess, setError, onAutoSave }) => {
  onAutoSave(val)
    .then(() => {
      setSaving(false);
      setSuccess(true);
    })
    .catch(err => {
      setSaving(false);

      if (hasErrorShape(err)) {
        err = transformErrorShape(err);
      }
      setError((err && err.message) || 'Something went wrong.');
    });
}, 800);

const Input = memo(
  ({
    disabled,
    placeholder = '',
    value = '',
    className,
    type = 'text',
    onChange,
    icon,
    autoFocus,
    onSubmit,
    rawOnChange,
    clear,
    spellCheck,
    onAutoSave,
    hasError = null,
    hasSuccess = false,
    isSaving = false,
    style,
    multiLine,
    maxLength,
    fluid,
    ...rest
  }) => {
    const [saving, setSaving] = useState(false);
    const [success, setSuccess] = useState(false);
    const [error, setError] = useState(null);
    hasError = hasError || error;
    hasSuccess = hasSuccess || success;
    isSaving = isSaving || saving;

    const inputEl = useRef(null);
    useEffect(() => {
      let currentInputElem = inputEl && inputEl.current;
      let timeout = null;
      if (currentInputElem && autoFocus) {
        timeout = setTimeout(() => inputEl.current.focus(), 50);
      }
      return () => {
        if (currentInputElem) {
          currentInputElem.blur();
        }
        if (timeout) {
          clearTimeout(timeout);
        }
      };
    }, [inputEl, autoFocus]);

    const debouncedAutoSave = onAutoSave
      ? value => {
          setSaving(true);
          setError(null);
          setSuccess(false);

          debounced(value, { onAutoSave, setSaving, setError, setSuccess });
        }
      : null;

    const props = {
      spellCheck,
      autoFocus,
      type,
      value: value || '',
      onChange: event => {
        let value = event.target.value;
        value = value.replace('\n', '');
        if (onChange) {
          onChange(value);
        }
        if (rawOnChange) {
          rawOnChange(event);
        }
        if (debouncedAutoSave) {
          debouncedAutoSave(value);
        }
      },
      onKeyDown: event => {
        if (onSubmit && event.key === 'Enter') {
          event.stopPropagation();
          event.preventDefault();
          return onSubmit();
        }
      },
      onBlur: () => {
        setSuccess(false);
      },
      placeholder,
      disabled,
      ...rest
    };

    const classes = cx(
      css.formInput,
      className,
      { [css.hasIcon]: icon },
      { [css.clear]: clear },
      { [css.hasError]: hasError },
      { [css.hasSuccess]: hasSuccess }
    );

    return (
      <>
        <div className={classes} style={style}>
          {icon && <Icon icon={icon} className={css.formInputIcon} size={16} name={icon} />}
          {multiLine ? (
            <TextareaAutosize
              maxLength={maxLength}
              {...multiLine}
              inputRef={inputEl}
              className={css.formInputInput}
              {...props}
            />
          ) : (
            <input ref={inputEl} className={css.formInputInput} {...props} />
          )}
          {isSaving && <Loader size={16} className={css.formInputIcon} />}
          {hasError && (
            <Icon size={16} color={'#ff7269'} icon="close" className={css.formInputIcon} />
          )}
          {hasSuccess && (
            <Icon size={16} color={'#5fe9b7'} icon="checkmark" className={css.formInputIcon} />
          )}
          {maxLength && multiLine ? (
            <span className={css.characterLimit}>
              {value.length}/{maxLength}
            </span>
          ) : null}
        </div>
        {hasError && <p className={css.inlineError}>{hasError}</p>}
      </>
    );
  }
);

export default Input;
