import React, { PureComponent } from 'react';
import { isRedirect } from '@reach/router';
import { track } from 'scripts/analytics';
import { Sender } from 'classes/Logs';

import { Button } from 'bebo-ui';

import './ErrorBoundary.module.scss';

const isChunkLoadError = error => {
  let message = (error || {}).message || '';
  return Boolean(message.match(/(Loading chunk )\d( failed.)/gi));
};

export default class ErrorBoundary extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { hasError: false, actingUser: {} };
  }

  componentDidCatch(error, errorInfo) {
    if (isRedirect(error)) {
      throw error;
    } else if (isChunkLoadError(error)) {
      console.log('*** chunkloaderror', error, errorInfo);

      //TODO put in different UI for these kinds of errors
      this.setState({ hasError: true });
    } else {
      const { message, stack } = error;
      const { componentStack } = errorInfo;

      this.setState({ hasError: true });
      const label = this.props.label || 'critical';

      const data = {
        log_level_tx: 'error',
        message_sa: message,
        stack_sa: stack,
        componentStack_sa: componentStack,
        type_tx: 'boundary'
      };

      track('error', 'caught', label, data);

      Sender.addRaw(data);
    }
  }

  render() {
    const { hasError } = this.state;
    const { children, fallback, isSilent } = this.props;
    if (hasError) {
      if (isSilent) {
        return null;
      }
      return fallback ? fallback : <ErrorMessage />;
    }
    return children;
  }
}

const ErrorMessage = () => {
  const reloadFn = () => {
    window.location.reload(true);
  };
  return (
    <div className="error-message">
      <div className="error-message__reason">
        <img
          alt="error"
          className="error-message__reason--image"
          src="https://a.imgdropt.com/image/68d12a49-0342-4312-9625-7e647cf280c7"
        />
        <span className="error-message__reason--text">Uh Oh!</span>
      </div>
      <span className="error-message__text">
        Sorry, it looks like something didn't go as expected. You can try to continue or let us know
        what happened in our discord.
      </span>
      <div className="error-message__actions">
        <Button onClick={reloadFn} width={300} className="error-message__action" primary>
          Try to continue
        </Button>
        <Button
          width={300}
          className="error-message__action"
          theme="discord"
          secondary
          link="https://discord.gg/bDYRvjb"
        >
          Find us in Discord
        </Button>
      </div>
    </div>
  );
};
