import React, { memo, useCallback, useState, useEffect } from 'react';
// import cx from 'classnames';

import { Link } from '@reach/router';
import useSnack from 'hooks/useSnack';
import useEntity from 'hooks/useEntity';
import MatchController from 'controllers/match';

import ActionCard from './card';

import { Button } from 'bebo-ui';
import Sheet from 'components/modal-sheet';

import useSocketSubscription from 'hooks/useSocketSubscription';
import useActingUser from 'hooks/useActingUser';

import css from './action-card.module.scss';

import DownloadPrompt from 'components/download-prompt';

const CreateTeamActionCard = memo(({ track, teamId, leagueId, cardRef }) => {
  const actingUser = useActingUser();
  const league = useEntity('league', leagueId);
  const snack = useSnack();
  const [sheet, setSheet] = useState('');

  const [userStatePc] = useSocketSubscription(`/user/state/pc`, {}, null, {
    isSingleValue: true,
    reFetch: true
  });

  const [requesting, setRequesting] = useState(false);

  const onRequestMatch = useCallback(() => {
    if ((!userStatePc || !userStatePc.exist) && !actingUser.mixer_username) {
      track('match_request_no_app', { userStatePc });
      setSheet('pc_app');
      return;
    }
    track('match_request');
    setRequesting(true);
    return MatchController.request(leagueId, teamId)
      .then(response => {
        if (response && response.result && response.result[0]) {
          track('match_request_success');
          return Promise.resolve();
        }
        setRequesting(false);
        console.log('match request response', response);
        return Promise.resolve();
      })
      .catch(err => {
        setRequesting(false);
        track('match_request_fail');
        console.error('match request error', err);
        snack.onError(err);
        return Promise.resolve();
      });
  }, [actingUser.mixer_username, leagueId, snack, teamId, track, userStatePc]);

  useEffect(() => {
    if (sheet === 'pc_app' && userStatePc && userStatePc.exist) {
      setSheet('');
      onRequestMatch();
    }
  }, [onRequestMatch, sheet, userStatePc]);

  const activeMatchId = actingUser.active_match_id;
  const hasMatchInfo = true;

  const hasActiveTournament = Boolean(league && league.active_tournament_id);

  const actionBtn = activeMatchId ? (
    <Link
      className={css.gamePlayButton}
      style={{ opacity: hasMatchInfo ? null : 0, transition: 'opacity 150ms' }}
      to={`/app/league/${leagueId}/match/${activeMatchId}`}
    >
      Return to Match
    </Link>
  ) : (
    <Button
      className={css.gamePlayButton}
      style={{ opacity: hasMatchInfo ? null : 0, transition: 'opacity 150ms' }}
      primary
      fluid
      link={
        hasActiveTournament
          ? `/app/league/${leagueId}/tournament/${league && league.active_tournament_id}`
          : null
      }
      onClick={hasActiveTournament ? null : onRequestMatch}
      loading={requesting}
    >
      {hasActiveTournament ? 'Go To Tournament' : 'Play Practice Match'}
    </Button>
  );

  return (
    <>
      <ActionCard
        title="Try your skills!"
        subTitle="You can play an unlimited number of practice matches. They do not influence anything related to tournaments."
        sticky
        className={css.teamCard}
        cardRef={cardRef}
        actionBtn={actionBtn}
      />

      <Sheet open={sheet === 'pc_app'} onClose={() => setSheet('')}>
        <DownloadPrompt />
      </Sheet>
    </>
  );
});

export default CreateTeamActionCard;
