import React, { memo, useState } from 'react';
import css from './platform.module.scss';

import { PUT } from 'classes/socket';

import { Button, Icon } from 'bebo-ui';

import Sheet from 'components/modal-sheet';

import config from 'scripts/config';

import useTrack from 'hooks/useTrack';

import { useMedia } from 'the-platform';

// import UserController from 'controllers/user';

const Platform = memo(({ onNext }) => {
  const [pickedPlatform, setPickedPlatform] = useState(null);
  const [processing, setProcessing] = useState(false);
  const track = useTrack(
    'signup_funnel',
    pickedPlatform ? `platform_${pickedPlatform}` : null,
    null,
    true
  );

  const handlePlatformSelect = platform => {
    setPickedPlatform(platform);
    setTimeout(() => track('click', { platform, label: `platform_${platform}` }));
  };

  const handleNext = () => {
    setProcessing(true);
    const payload = { [`has_${pickedPlatform === 'win' ? 'pc' : pickedPlatform}`]: true };
    return PUT('/user', payload)
      .then(response => {
        setProcessing(false);
        onNext();
      })
      .catch(err => {
        console.log('*** put user platform err', err);
        setProcessing(false);
        onNext();
      });
  };

  return (
    <>
      <div>
        <div className={css.cards}>
          <Button className={css.cardItem} onClick={() => handlePlatformSelect('win')}>
            <Icon icon="win10" size={40} />
            <h4>Windows 10</h4>
          </Button>
          <Button className={css.cardItem} onClick={() => handlePlatformSelect('ps4')}>
            <Icon icon="ps4" size={40} />
            <h4>PS4</h4>
          </Button>
          <Button className={css.cardItem} onClick={() => handlePlatformSelect('xbox')}>
            <Icon icon="xbox" size={40} />
            <h4>XBOX</h4>
          </Button>
        </div>
      </div>
      <Sheet open={pickedPlatform} onClose={() => setPickedPlatform(null)}>
        {pickedPlatform === 'ps4' ? (
          <NotYetSheet
            platform={pickedPlatform}
            track={track}
            onNext={handleNext}
            processing={processing}
          />
        ) : pickedPlatform === 'xbox' ? (
          <NotYetSheet
            platform={pickedPlatform}
            track={track}
            onNext={handleNext}
            processing={processing}
          />
        ) : (
          <DownloadNowSheet track={track} onNext={handleNext} processing={processing} />
        )}
      </Sheet>
    </>
  );
});

export default Platform;

const DownloadNowSheet = memo(({ track, onNext, processing }) => {
  const noMobile = useMedia({ minWidth: 1024 }, window.innerWidth > 1024);
  return (
    <div className={css.downloadSheet}>
      <h4>Download the app to play</h4>
      <div className={css.downloadSheetInner}>
        {!noMobile ? (
          <>
            <h5>On your PC, go to: </h5>
            <Button
              fluid
              primary
              link={`${config.WWW_URL}/download`}
              external
              onClick={() => {
                track('download_link', { platform: 'win' });
              }}
            >
              bebo.com/download
            </Button>
            <p>Then log in & play!</p>
          </>
        ) : (
          <Button
            primary
            fluid
            link={config.APP_DL_URL}
            external
            loading={processing}
            onClick={() => {
              track('download', { platform: 'win' });
              onNext();
            }}
          >
            <Icon icon="win10" size={40} /> Download Windows 10 App
          </Button>
        )}
      </div>
      <p>Already have the app?</p>
      <Button
        secondary
        fluid
        loading={processing}
        onClick={() => {
          track('continue', { platform: 'win10' });
          onNext();
        }}
      >
        Continue
      </Button>
    </div>
  );
});

// const XboxSheet = memo(({ platform, track, onNext, processing }) => {
//   const [mixer_username, setMixer_username] = useState('');
//   const [loading, setLoading] = useState('');
//   const [error, setError] = useState(null);

//   const saveUserName = () => {
//     setLoading(true);
//     setError(null);
//     track('mixer_submit', { mixer_username });
//     return UserController.update({ mixer_username })
//       .then(response => {
//         setLoading(false);
//         onNext();
//         track('mixer_success', { mixer_username });
//         console.log('*** user update repsonse', { response });
//       })
//       .catch(err => {
//         setLoading(false);
//         console.error('*** user update err', err);
//         setError(err.message || 'Something went wrong, please try again.');
//       });
//   };

//   return (
//     <div className={css.notYetSheet}>
//       <h2>
//         <Icon icon={platform} viewBox="0 0 50 50" size={40} />
//       </h2>
//       <h4>In order to play on xbox you need to stream to mixer</h4>
//       <p>That’s how we track and verify your scores live</p>
//       <h5>What is your mixer username?</h5>
//       <Input value={mixer_username} onChange={setMixer_username} onSubmit={saveUserName} />

//       <Button
//         primary={mixer_username}
//         secondary={!mixer_username}
//         fluid
//         loading={processing || loading}
//         onClick={() => {
//           if (!mixer_username) {
//             track('skip', { platform });
//             onNext();
//           } else {
//             saveUserName();
//           }
//         }}
//       >
//         {mixer_username ? 'Save' : 'Skip'}
//       </Button>
//       {error && <p>{error}</p>}
//     </div>
//   );
// });

const NotYetSheet = memo(({ platform, track, onNext, processing }) => {
  return (
    platform && (
      <div className={css.notYetSheet}>
        <h4>
          <Icon icon={platform} size={40} /> {(platform || '').toUpperCase()} coming soon
        </h4>
        <p>Sorry, Bebo is currently only available on Windows 10.</p>
        <p>You can still join the league and set up your team.</p>
        <Button
          primary
          fluid
          loading={processing}
          onClick={() => {
            track('continue', { platform });
            onNext();
          }}
        >
          Continue
        </Button>
      </div>
    )
  );
});
