import React, { memo, useRef, useEffect, useCallback } from 'react';

import PropTypes from 'prop-types';

import cx from 'classnames';
import css from './hls-video.module.scss';

import Hls from 'hls.js';

const HLSVideo = ({ url, className = '', hlsOptions = {}, onEnd, onError, ...rest }) => {
  const videoRef = useRef(null);
  const _onError = useCallback(
    err => {
      const error = err instanceof Error ? err : new Error(err);
      console.error('HLSVideo Error', error);
      if (onError) {
        onError(error);
      }
    },
    [onError]
  );
  const _onEnd = useCallback(
    e => {
      if (onEnd) {
        onEnd(e);
      }
      console.log('playback stopped', e);
    },
    [onEnd]
  );
  useEffect(() => {
    let videoElem = videoRef && videoRef.current;

    let hls = null;
    if (videoElem) {
      if (Hls.isSupported()) {
        hls = new Hls({ maxBufferLength: 10, liveSyncDurationCount: 2 });
        hls.loadSource(url);
        hls.attachMedia(videoElem);
        hls.on(Hls.Events.MANIFEST_PARSED, () => {
          if (rest.autoPlay) {
            //handle play promise via: https://goo.gl/LdLk22
            let playPromise = videoElem.play();
            if (playPromise !== undefined) {
              playPromise.catch(err => {
                console.log('failed to auto play video', err);
              });
            }
          }
        });
        hls.on(Hls.Events.ERROR, function(event, data) {
          if (data.fatal) {
            switch (data.type) {
              case Hls.ErrorTypes.NETWORK_ERROR:
                // try to recover network error
                console.log('fatal network error encountered, try to recover');
                hls.startLoad();
                break;
              case Hls.ErrorTypes.MEDIA_ERROR:
                console.log('fatal media error encountered, try to recover');
                hls.recoverMediaError();
                break;
              default:
                // cannot recover
                hls.destroy();
                break;
            }
            _onError('fatal hls error');
          }
        });
      } else if (videoElem.canPlayType('application/vnd.apple.mpegurl')) {
        videoElem.src = url;
        videoElem.addEventListener('loadedmetadata', function() {
          if (rest.autoPlay) {
            //handle play promise via: https://goo.gl/LdLk22
            let playPromise = videoElem.play();
            if (playPromise !== undefined) {
              playPromise.catch(err => {
                console.log('failed to auto play video', err);
              });
            }
          }
        });
      } else {
        _onError('hls.js not supported');
      }
      videoElem.addEventListener('ended', _onEnd);
    }

    return () => {
      if (videoElem) {
        videoElem.removeEventListener('ended', _onEnd);
      }
      if (hls) {
        hls.destroy();
      }
    };
  }, [videoRef, url, _onEnd, rest.autoPlay, _onError]);

  return <video className={cx(css.video, className)} ref={videoRef} {...rest} />;
};

HLSVideo.propTypes = {
  url: PropTypes.string.isRequired,
  hlsOptions: PropTypes.object
};

export default memo(HLSVideo);
