import React, { memo, useEffect, useMemo } from 'react';
import cx from 'classnames';
import css from './leaderboard-item.module.scss';

import { TeamAvatar, Icon } from 'bebo-ui';

import useSubscribeTo from 'hooks/useSubscribeTo';
import useEntity from 'hooks/useEntity';

const LeaderboardItem = memo(
  ({
    isMe,
    isActive,
    team,
    onClick,
    topTeam,
    style,
    rank,
    win = 0,
    loss = 0,
    draw = 0,
    points = 0,
    qualified
  }) => {
    const { team_id } = team;
    let entityTeam = {};
    let teamError = null;
    if (isMe) {
      const [tError] = useSubscribeTo('/team', { team_id });
      teamError = tError;
      entityTeam = useEntity('team', team_id);
    }

    team = useMemo(() => (isMe ? { ...team, ...entityTeam } : team), [isMe, team, entityTeam]);

    let { silver_cnt, gold_cnt } = team;
    silver_cnt = parseInt(silver_cnt || 0, 10);
    gold_cnt = parseInt(gold_cnt || 0, 10);

    useEffect(() => {
      if (onClick && isActive && team) {
        team.qualified = qualified;
        onClick(team);
      }
    }, [onClick, isActive, team, qualified]);

    useEffect(() => {
      if (teamError) {
        console.error('failed to subscribe to team', { team_id, error: teamError });
      }
    }, [teamError, team_id]);
    return (
      <div
        style={style}
        onClick={() => onClick(team)}
        className={cx(css.container, { [`${css.isMe}`]: isMe })}
      >
        <span className={css.rank}>{rank}</span>
        <div className={css.user}>
          <TeamAvatar {...team} size={30} />
          <span className={css.names}>
            <span className={css.teamname}>
              {team.name}
              {gold_cnt ? (
                <span className={cx(css.tournament_placement, css.gold)}>
                  <Icon icon="line-trophy" size={13} />
                  <span>x{gold_cnt}</span>
                </span>
              ) : null}
              {silver_cnt ? (
                <span className={cx(css.tournament_placement, css.silver)}>
                  <Icon icon="line-trophy" size={13} />
                  <span>x{silver_cnt}</span>
                </span>
              ) : null}
            </span>
            <span className={css.usernames}>
              {team.users.map((u, i) => (
                <span key={u.user_id}>
                  <span>{u.username}</span>
                  {i < team.users.length - 1 && <span> / </span>}
                </span>
              ))}
            </span>
          </span>
        </div>
        {qualified && <Icon icon="ticket" size={23} className={css.qualified} />}
        <span className={css.score}>
          <b>{points}</b>
          <span>pts</span>
        </span>
      </div>
    );
  }
);

export default LeaderboardItem;
