import React, { memo, useCallback, useState, useMemo, useEffect } from 'react';

import { colors } from '../../theme.js';

import { Icon, Button, Column } from 'bebo-ui';
import { View } from 'components/view';
import NavBar from 'components/nav-bar';

import css from './lobby.module.scss';

import LobbyHeader from './lobby-header';
import LobbyTeam from './lobby-team';

import LobbyTracking from './lobby-tracking';

import Sheet from 'components/modal-sheet';
import MatchInfoSheet from 'components/match/match-info-sheet';

import useUserSettings from 'hooks/useUserSettings';

const teamSizeToName = { 1: 'Solos', 2: 'Duos', 3: 'Trios', 4: 'Squads' };

const Lobby = memo(
  ({
    lobbyState,
    league,
    redirectUrl,
    ownUserState,
    actingTeam,
    canCancel,
    isTournamentMatch,
    userStates,
    onReady,
    onUnReady,
    onQuitMatch,
    matchRounds
  }) => {
    const [sheet, setSheet] = useState('');
    const onMatchInfo = useCallback(() => {
      setSheet('match_info');
    }, []);

    const modeName = useMemo(() => {
      if (league && league.team_size) {
        return teamSizeToName[league.team_size];
      }
      return '';
    }, [league]);

    // const [isHeaderVisible] = useState(true);
    // const { ready, pc, xbox, bran } = ownUserState;
    // console.log('lobbyState', lobbyState);

    const nav = (
      <NavBar
        transparent={true}
        background={colors.surfaceOne}
        description={isTournamentMatch ? 'Tournament Match Lobby' : 'League Match Lobby'}
        leftLink={redirectUrl}
        leftAction={onQuitMatch}
        rightIcon={
          <Icon color="#fff" icon="info" onClick={onMatchInfo} style={{ marginRight: 5 }} />
        }
        absolute
        hideTabs
      />
    );

    const footer = (
      <LobbyFooter
        lobbyState={lobbyState}
        ownUserState={ownUserState}
        onReady={onReady}
        onUnReady={onUnReady}
        modeName={modeName}
      />
    );

    return (
      <>
        <View nav={nav} footer={footer}>
          <LobbyHeader
            matchRounds={matchRounds}
            modeName={modeName}
            isTournamentMatch={isTournamentMatch}
            league={league}
          />
          <LobbyTeam actingTeam={actingTeam} userStates={userStates} />
        </View>
        <Sheet open={sheet === 'match_info'} onClose={() => setSheet('')}>
          <MatchInfoSheet modeName={modeName} />
        </Sheet>
      </>
    );
  }
);

export default Lobby;

const ACCEPTED_STREAM_KEY = 'user_accepted_stream';

const LobbyFooter = memo(({ lobbyState, ownUserState, onReady, onUnReady, modeName }) => {
  const isReady = lobbyState !== 'readyUp';

  const [acceptedStream, setAcceptedStream] = useUserSettings(ACCEPTED_STREAM_KEY);
  const [open, setOpen] = useState(false);
  const [ready, setReady] = useState(false);

  useEffect(() => {
    if (!isReady && ready && acceptedStream) {
      onReady();
      setOpen(false);
    } else if (isReady && !ready) {
      onUnReady();
    }
  }, [ready, acceptedStream, isReady, onReady, onUnReady]);

  const _onReady = () => {
    setReady(true);
    if (!acceptedStream) {
      setOpen(true);
    }
  };

  const _accept = () => {
    setAcceptedStream(true);
  };

  return (
    <>
      <div className={css.footer}>
        {isReady ? (
          <LobbyTracking lobbyState={lobbyState} ownUserState={ownUserState} modeName={modeName} />
        ) : null}
        <div className={css.lobbyAction}>
          <Button
            className={css.lobbyReadyButton}
            primary
            fluid
            loading={open}
            disabled={open}
            onClick={isReady ? () => setReady(false) : _onReady}
          >
            {isReady ? 'UNREADY' : `I'm ready to play!`}
            {!isReady && <div className={css.shine} />}
          </Button>
        </div>
      </div>
      <Sheet open={open} onClose={() => setOpen(false)}>
        <Column>
          <h2>Is this thing on?</h2>
          <p style={{ marginTop: 5 }}>
            During your match Bebo transmits a video and your audio stream of your gameplay. <br />
            We do this to automatically track your points and to let you show off your skills.
          </p>
        </Column>
        <Column style={{ marginTop: 20 }}>
          <Button primary onClick={_accept} fluid>
            Ok
          </Button>
          <Button secondary fluid onClick={() => setOpen(false)}>
            Cancel
          </Button>
        </Column>
      </Sheet>
    </>
  );
});
