import React, { useState, useMemo } from 'react';

import { Input, Button } from 'bebo-ui';

import { Redirect, Link } from '@reach/router';

import posed from 'react-pose';

import UserController, { isLoggedIn$ } from 'controllers/user';

import css from './login.module.scss';

import useTrack from 'hooks/useTrack';

import ReactPhoneInput from 'react-phone-input-2';

import useObservable from 'hooks/useObservable';

const Stepper = posed.div({
  visible: {
    x: ({ step }) => `${-100 * step}%`,
    transition: {
      x: { type: 'spring', stiffness: 750, damping: 50 }
    }
  },
  props: { step: 0 }
});

const Step = posed.div({
  open: {
    opacity: 1
  },
  closed: {
    opacity: 0
  }
});

const LoginFlow = ({ isSignup, track, allowUserName, starRoute, isEmbedded }) => {
  const isLoggedIn = useObservable(isLoggedIn$);
  const iTrack = useTrack('login_flow');
  const [useUserName, setUseUserName] = useState(false);
  const [step, setStep] = useState(0);
  const [phoneNumber, setPhoneNumber] = useState('');
  const [username, setUsername] = useState('');
  const [challengeCode, setChallengeCode] = useState('');
  const [challenge, setChallenge] = useState({});
  const [password, setPassword] = useState('');
  const [error, setError] = useState(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [consent, setConsent] = useState(false);

  const validChallengeCode = useMemo(() => {
    return useUserName &&
      !(challenge.require && challenge.require[0] && challenge.require[0] === 'pin')
      ? password && password.length
      : `${challengeCode}` && `${challengeCode}`.length === 6;
  }, [challengeCode, useUserName, password, challenge.require]);

  if (isLoggedIn === true) {
    return <Redirect noThrow to={starRoute ? starRoute : '/join'} />;
  } else if (isLoggedIn !== false) {
    return null;
  }

  const handleTrack = (action, data) => {
    if (track) {
      track(action, data);
    }
    iTrack(action, data);
  };

  const handleSubmitStepOne = ({ user_name, phone_number }) => {
    setError(null);
    if (!consent && isSignup) {
      setError('Please accept the Terms of Service and Privacy Policy');

      return;
    }
    setIsSubmitting(true);

    let prom = isSignup
      ? UserController.signup(null, user_name || null, null, phone_number || null)
      : UserController.login(user_name || null, null, phone_number, null);

    return prom
      .then(response => {
        setChallenge(response);
        setIsSubmitting(false);
        if (step === 0) {
          setStep(step + 1);
        }
      })
      .catch(err => {
        setError((err && err.message) || 'Something went wrong, please try again.');
        setIsSubmitting(false);
      });
  };

  const handleSubmitStep2 = ({ password, pin }) => {
    setError(null);
    if (!consent && isSignup) {
      setError('Please accept the Terms of Service and Privacy Policy');

      return;
    }
    setIsSubmitting(true);

    let prom = isSignup
      ? UserController.signup(
          null,
          username || null,
          password || null,
          phoneNumber || null,
          pin || null
        )
      : UserController.login(username || null, password || null, phoneNumber || null, pin || null);

    return prom
      .then(response => {
        setIsSubmitting(false);
        handleTrack('success');
      })
      .catch(err => {
        setError((err && err.message) || 'Something went wrong, please try again.');
        setIsSubmitting(false);
        setPassword('');
        setChallengeCode('');
        handleTrack('fail', { error_tx: (err && err.message) || '' });
      });
  };

  const handleConsentCheck = () => {
    setConsent(!consent);
  };

  return (
    <>
      <Stepper poseKey={step} className={css.wrapper} step={step} pose={'visible'}>
        <Step className={css.container} pose={step === 0 ? 'open' : 'closed'}>
          <label>
            Enter your {useUserName ? 'username' : 'phone number'} to {isSignup ? 'join' : 'log in'}
          </label>
          <div className={css.inputWrapper}>
            {useUserName ? (
              <Input
                autoFocus={step === 0}
                placeholder="username"
                value={username}
                onChange={setUsername}
                onSubmit={() => {
                  handleTrack('submit_username', { username });
                  return handleSubmitStepOne({ user_name: username });
                }}
              />
            ) : (
              <ReactPhoneInput
                autoFocus={step === 0}
                value={phoneNumber}
                disableAreaCodes
                preferredCountries={['us', 'ca', 'gb']}
                defaultCountry="us"
                onKeyDown={event => {
                  if (event.key === 'Enter') {
                    event.stopPropagation();
                    event.preventDefault();
                    handleTrack('submit_number', { phone_number: phoneNumber });
                    return handleSubmitStepOne({ phone_number: phoneNumber });
                  }
                }}
                onChange={(number, meta) => {
                  let cleanedNumber = number.replace(/[^\d|+]/g, '');
                  setPhoneNumber(cleanedNumber);
                }}
              />
            )}
          </div>
          {isSignup ? (
            <small className="signin__main__small" style={{ margin: '10px 0' }}>
              <label className="container">
                <input checked={consent} onChange={handleConsentCheck} type="checkbox" />
                <span className="checkmark" />
              </label>
              <small>
                {' '}
                I am over the age of 13 and I consent to the Bebo{' '}
                <a target="_blank" href="/legal/terms" rel="noopener noreferrer">
                  Terms of Service
                </a>{' '}
                and{' '}
                <a target="_blank" href="/legal/privacy" rel="noopener noreferrer">
                  {' '}
                  Privacy Policy
                </a>
              </small>
            </small>
          ) : null}
          <Button
            loading={isSubmitting}
            primary
            fluid
            onClick={() => {
              if (useUserName) {
                handleTrack('submit_username', { username });
              } else {
                handleTrack('submit_number', { phone_number: phoneNumber });
              }

              handleSubmitStepOne({ phone_number: phoneNumber, user_name: username });
            }}
          >
            {isSignup ? 'Join' : 'Log In'}
          </Button>
          {!isSignup && allowUserName && (
            <Button
              secondary
              fluid
              onClick={() => {
                handleTrack('switch_mode', { useUserName });
                setUsername('');
                setChallengeCode('');
                setPhoneNumber('');
                setPassword('');
                setUseUserName(!useUserName);
              }}
            >
              {useUserName ? 'or use phone number' : 'or use username'}
            </Button>
          )}
          {!isEmbedded && (
            <>
              {isSignup ? (
                <small className="signin__main__small">
                  <span>Already have an account?</span>{' '}
                  <Link
                    to={starRoute ? `/login/to${starRoute}` : '/login'}
                    className={css.link}
                    onClick={() => {
                      handleTrack('navigate', { to: 'login' });
                    }}
                  >
                    Log in
                  </Link>
                </small>
              ) : (
                <small className="signin__main__small">
                  <span>Don't have an account?</span>{' '}
                  <Link
                    to={starRoute ? `/signup/to${starRoute}` : '/signup'}
                    className={css.link}
                    onClick={() => {
                      handleTrack('navigate', { to: 'signup' });
                    }}
                  >
                    Sign up
                  </Link>
                </small>
              )}
            </>
          )}
        </Step>
        <Step className={css.container} pose={step === 1 ? 'open' : 'closed'}>
          <label>{challenge.message}</label>
          <div className={css.inputWrapper}>
            {challenge.require && challenge.require[0] && challenge.require[0] === 'pin' ? (
              <Input
                fluid
                autoFocus={step === 1}
                value={challengeCode}
                onChange={setChallengeCode}
                type="text"
                placeholder="123456"
                maxLength={6}
                autoComplete="one-time-code"
                id="single-factor-code-text-field"
                onSubmit={
                  validChallengeCode
                    ? () => {
                        handleTrack('submit_pin', { phone_number: phoneNumber });
                        handleSubmitStep2({ password, pin: challengeCode });
                      }
                    : null
                }
              />
            ) : (
              <Input
                fluid
                autoFocus={step === 1}
                value={password}
                onChange={setPassword}
                type="password"
                placeholder="******"
                onSubmit={
                  validChallengeCode
                    ? () => {
                        handleTrack('submit_password', { username });
                        handleSubmitStep2({ password });
                      }
                    : null
                }
              />
            )}
          </div>

          {validChallengeCode ? (
            <Button
              loading={isSubmitting}
              primary
              fluid
              onClick={() => {
                if (challenge.require && challenge.require[0] && challenge.require[0] === 'pin') {
                  handleTrack('submit_pin', { phone_number: phoneNumber });
                } else {
                  handleTrack('submit_password', { username });
                }

                handleSubmitStep2({ password, pin: challengeCode });
              }}
            >
              {useUserName &&
              !(challenge.require && challenge.require[0] && challenge.require[0] === 'pin')
                ? 'Submit'
                : 'Verify'}
            </Button>
          ) : useUserName &&
            !(challenge.require && challenge.require[0] && challenge.require[0] === 'pin') ? (
            <Button
              secondary
              fluid
              onClick={() => {
                handleTrack('back', { username });
                setStep(step - 1);
              }}
            >
              Go back
            </Button>
          ) : (
            <>
              <Button
                loading={isSubmitting}
                secondary
                fluid
                onClick={() => {
                  handleTrack('resend', { phone_number: phoneNumber });
                  handleSubmitStepOne({ phone_number: phoneNumber });
                }}
              >
                re - send the pin
              </Button>
              <Button
                className={css.back}
                secondary
                fluid
                onClick={() => {
                  handleTrack('back', { username });
                  setStep(step - 1);
                }}
              >
                Go back
              </Button>
            </>
          )}
        </Step>
      </Stepper>
      <h5 className={css.errorMessage} style={{ opacity: error ? 1 : 0 }}>
        {error || 'error'}
      </h5>
    </>
  );
};

export default LoginFlow;
