import React, { memo, useMemo, useState, useCallback, useEffect } from 'react';
import { Column, TeamAvatar, Row, Icon, Button } from 'bebo-ui';

// import Linkify from 'react-linkify';
import titlelyfy from 'title';

// import HLSVideo from 'components/hls-video';

import css from './match.module.scss';

import { isNumber } from 'scripts/utils';
import cx from 'classnames';

import { colors } from '../../theme';
import { isDesktopApp, isAppVersionBigger } from 'scripts/utils';

import NavBar from 'components/nav-bar';
import Sheet from 'components/modal-sheet';
// import Section from 'components/section';

import MatchInfoSheet from './match-info-sheet';
import MatchEndedAction from './match-ended-action';

import MatchTeams from './match-teams';
import MatchVideo from './match-video';

import { winnerQuotes, loserQuotes } from './quotes.js';

import unkownOpponentImg from './assets/unknown-opponent.png';

const checkVideoCapability = () => {
  if (!isDesktopApp()) {
    return true;
  }
  if (isAppVersionBigger('2.0.154')) {
    return true;
  }
  return false;
};

console.log('*** isAppVersionBigger', isAppVersionBigger('2.0.154'));

const teamSizeToName = { 1: 'Solos', 2: 'Duos', 3: 'Trios', 4: 'Squads' };

const matchIdToWinnerQuote = new Map();
const matchIdToLoserQuote = new Map();

const getRandomWinnerQuote = match_id => {
  if (matchIdToWinnerQuote.has(match_id)) {
    return matchIdToWinnerQuote.get(match_id);
  }
  const quote = winnerQuotes[Math.floor(Math.random() * winnerQuotes.length)];
  matchIdToWinnerQuote.set(match_id, quote);
  return quote;
};

const getRandomLoserQuote = match_id => {
  if (matchIdToLoserQuote.has(match_id)) {
    return matchIdToLoserQuote.get(match_id);
  }
  const quote = loserQuotes[Math.floor(Math.random() * loserQuotes.length)];
  matchIdToLoserQuote.set(match_id, quote);
  return quote;
};

const Match = memo(
  ({
    currentRound,
    league,
    onQuitMatch,
    redirectUrl,
    isOwnMatch,
    sets,
    isActiveMatch,
    ownUserState,
    userStates,
    canCancel,
    canBack,
    matchRounds = 3,
    actingTeam,
    isTournamentMatch,
    track,
    matchState,
    matchOutcome
  }) => {
    const leftCurrentRound = useMemo(() => {
      if (isActiveMatch) {
        return currentRound;
      }
      return (sets[0] && sets[0].round) || 0;
    }, [isActiveMatch, currentRound, sets]);

    const rightCurrentRound = useMemo(() => {
      return (sets[1] && sets[1].round) || 0;
    }, [sets]);

    const leftScores = useMemo(() => {
      return ((sets[0] && sets[0].scores) || []).slice(0, leftCurrentRound);
    }, [leftCurrentRound, sets]);
    const rightScores = useMemo(() => {
      return ((sets[1] && sets[1].scores) || []).slice(
        0,
        isActiveMatch ? Math.max(leftCurrentRound - 1, 0) : rightCurrentRound
      );
    }, [isActiveMatch, leftCurrentRound, rightCurrentRound, sets]);

    const leftScore = useMemo(() => {
      return leftScores.reduce((a, b) => a + b, 0);
    }, [leftScores]);
    const rightScore = useMemo(() => {
      return rightScores.reduce((a, b) => a + b, 0);
    }, [rightScores]);

    const leftRounds = useMemo(() => {
      let rounds = [];
      for (let i = 0; i < matchRounds; i++) {
        rounds[i] = isNumber(leftScores[i])
          ? { score: leftScores[i], active: i === leftCurrentRound - 1 }
          : { locked: false, active: i === leftCurrentRound - 1 };
      }
      return rounds;
    }, [leftCurrentRound, leftScores, matchRounds]);
    const rightRounds = useMemo(() => {
      let rounds = [];
      for (let i = 0; i < matchRounds; i++) {
        rounds[i] = isNumber(rightScores[i])
          ? { score: rightScores[i], active: i === rightCurrentRound - 1 }
          : { locked: isActiveMatch, active: i === rightCurrentRound - 1 };
      }
      return rounds;
    }, [isActiveMatch, matchRounds, rightCurrentRound, rightScores]);

    const [sheet, setSheet] = useState('');

    const onMatchInfo = useCallback(() => {
      setSheet('match_info');
    }, []);

    const modeName = useMemo(() => {
      if (league && league.team_size) {
        return teamSizeToName[league.team_size];
      }
      return '';
    }, [league]);

    const matchType = useMemo(() => {
      if (isTournamentMatch) {
        return `Tournament ${modeName} Match`;
      }
      return `${modeName} Match`;
    }, [modeName, isTournamentMatch]);

    const [selectedUser, _setSelectedUser] = useState(null);

    const setSelectedUser = useCallback(
      user => {
        if (matchState !== 'ended') {
          _setSelectedUser(user);
        }
      },
      [matchState]
    );

    useEffect(() => {
      if (!selectedUser) {
        for (let set of sets) {
          if (set && set.state === 'started' && set.team && set.team.users) {
            for (let user of set.team.users) {
              const userState = userStates[user.user_id];
              if (userState && userState.bran) {
                setSelectedUser(set.team.users[0]);
                break;
              }
            }
          }
        }
      }
    }, [selectedUser, setSelectedUser, sets, userStates]);

    const streamAvailable = checkVideoCapability() && selectedUser;
    return (
      <>
        <NavBar
          sticky
          absolute
          leftLink={redirectUrl}
          description={matchType}
          rightIcon={
            <Icon color="#fff" icon="info" onClick={onMatchInfo} style={{ marginRight: 5 }} />
          }
          hideTabs
        />
        <div style={{ paddingBottom: 80 }}>
          <MatchHeader sets={sets} leftScore={leftScore} rightScore={rightScore} />
          <MatchFloatingCard
            leftRounds={leftRounds}
            rightRounds={rightRounds}
            currentRound={currentRound}
            isOwnMatch={isOwnMatch}
            isOwnActiveMatch={isActiveMatch}
            matchState={matchState}
            matchOutcome={matchOutcome}
            sets={sets}
            matchRounds={matchRounds}
            modeName={modeName}
          />
          {streamAvailable && (
            <MatchVideo
              selectedUser={selectedUser}
              isActiveMatch={isActiveMatch}
              userStates={userStates}
              sets={sets}
            />
          )}
          <MatchTeams
            onSelectUser={setSelectedUser}
            selectedUser={selectedUser}
            sets={sets}
            actingTeam={actingTeam}
            isOwnActiveMatch={isActiveMatch}
            userStates={userStates}
          />
          {isActiveMatch ? (
            <MatchControls
              onQuitMatch={onQuitMatch}
              canCancel={canCancel}
              canBack={canBack}
              redirectUrl={redirectUrl}
              userState={ownUserState}
            />
          ) : isOwnMatch && !isTournamentMatch ? (
            <MatchEndedAction actingTeam={actingTeam} league={league} track={track} />
          ) : null}
        </div>
        <Sheet open={sheet === 'match_info'} onClose={() => setSheet('')}>
          <MatchInfoSheet modeName={modeName} />
        </Sheet>
      </>
    );
  }
);

export default Match;

const MatchControls = memo(({ onQuitMatch, canCancel, canBack, redirectUrl, userState }) => {
  const [isLeaving, setIsLeaving] = useState(false);
  const isReady = useMemo(() => Boolean(userState.bran), [userState.bran]);
  const message = isReady ? 'Tracking' : 'Not Tracking';
  return (
    <Row className={css.matchControls}>
      <div
        className={css.matchControlsStatus}
        style={{ background: isReady ? colors.colorGreen : colors.colorRed }}
      />
      <h4>{message} Game</h4>
      <Button
        loading={isLeaving}
        onClick={() => {
          setIsLeaving(true);
          return onQuitMatch().finally(() => {
            setIsLeaving(false);
          });
        }}
        secondary
      >
        {canBack ? 'Go Back' : canCancel ? 'Cancel Match' : 'Leave Match'}
      </Button>
    </Row>
  );
});

const MatchFloatingCard = memo(
  ({
    leftRounds,
    rightRounds,
    currentRound,
    matchState,
    matchOutcome,
    isOwnMatch,
    isOwnActiveMatch,
    sets,
    matchRounds,
    modeName
  }) => {
    const currentStatus = useMemo(() => {
      let status = 'info';
      let round = `Game ${currentRound}`;
      let title = '';
      let message = '';
      let icon = '';
      let color = '';
      let iconColor = '';
      const match_id = sets && sets[0] && sets[0].set_id;

      let winnerQuote = getRandomWinnerQuote(match_id);
      let loserQuote = getRandomLoserQuote(match_id);

      if (isOwnMatch) {
        let hasOpponent = Boolean(sets && sets[1] && sets[1].team);
        if (!hasOpponent) {
          if (currentRound < 1) {
            round = 'Play Now';
            title = `Start a new game of ${modeName}`;
            color = colors.primary;
            iconColor = colors.surfaceOne;
            message = `You do not have to wait for your opponent to show up.`;
            icon = 'checkmark';
          } else if (currentRound > matchRounds) {
            round = 'Ended';
            title = `All done`;
            message = `You finished your games. We'll notify you once your opponent starts playing.`;
            icon = 'loading';
          } else {
            title = `Play Game ${currentRound} / ${matchRounds}`;
            message = `At the end of each game you will automatically advance to the next game.`; //add info / countdown about timeout
            icon = 'checkmark';
          }
        } else {
          if (currentRound < 1) {
            round = 'Play Now';
            title = `Start a new game of ${modeName}`;
            message = `You can play at your own pace. The match ends once both teams finish all 3 games.`;
          } else if (currentRound > matchRounds) {
            let outcome = sets && sets[0] && sets[0].outcome;

            if (outcome === 'win') {
              status = 'success';
              round = 'Victory';
              title = `You won this match`;
              message = (
                <span className={css.quote}>
                  <span>“{winnerQuote.message}”</span>
                  <br />
                  <span>{winnerQuote.author}</span>
                </span>
              );

              icon = 'line-trophy';
            } else if (outcome === 'loss') {
              status = 'error';
              round = 'Defeat';
              title = `You lost this match`;
              message = (
                <span className={css.quote}>
                  <span>“{loserQuote.message}”</span>
                  <br />
                  <span>{loserQuote.author}</span>
                </span>
              );
              icon = 'close';
            } else if (outcome === 'draw') {
              round = 'Draw';
              title = `You tied`;
              message = `Have a participation trophy. ༼ つ ◕_◕ ༽つ 🏆`;
              icon = 'lock';
            } else {
              round = 'Ended';
              title = `All done`;
              message = `You finished your games. We'll notify you once your opponent finishes playing.`;
              icon = 'loading';
            }
          } else {
            title = `Play Game ${currentRound} / ${matchRounds}`;
            message = `At the end of each game you will automatically advance to the next game.`; //add info / countdown about timeout
          }
        }
      } else {
        //not own match
        if (matchState === 'ended') {
          let winner = (sets.find(set => set.outcome === 'win') || {}).team;
          round = 'Ended';
          if (winner) {
            status = 'success';
            title = `"${winner.name}" won!`;
            message = (
              <span className={css.quote}>
                <span>“{winnerQuote.message}”</span>
                <br />
                <span>{winnerQuote.author}</span>
              </span>
            );
            icon = 'line-trophy';
          } else {
            title = 'Draw';
            message = `This match was a tie. 	¯\\_(ツ)_/¯`;
            icon = 'lock';
          }
        } else if (matchState === 'started') {
          title = 'Ongoing Match';
          message = `Two teams enter. One team leaves. Find out soon.`;
          icon = 'checkmark';
        }
      }

      title = titlelyfy(title);

      return { status, round, title, message, icon, iconColor, color };
    }, [currentRound, matchState, isOwnMatch, sets, matchRounds, modeName]);

    const isActive = currentRound <= matchRounds;
    return (
      <Column className={css.matchFloatingCard}>
        <Row className={css.matchRounds}>
          <div>
            {leftRounds.map((round, i) => (
              <Round key={`round_${i}`} {...round} />
            ))}
          </div>
          <div>
            <h4 style={isActive ? { color: '#fdd251' } : {}}>{currentStatus.round}</h4>
          </div>
          <div>
            {rightRounds
              // .slice()
              // .reverse()
              .map((round, i) => (
                <Round key={`round_${i}`} {...round} reverse />
              ))}
          </div>
        </Row>
        <Row className={cx(css.matchStatus, css[currentStatus.status])}>
          {/* {currentStatus.icon && (
            <div className={css.matchStatusIcon} style={currentStatus.color ? {background:currentStatus.color} : null}>
              {currentStatus.icon === 'loading' ? (
                <Loader size={20} style={{ margin: 'auto' }} />
              ) : (
                <Icon icon={currentStatus.icon} style={{ margin: 'auto', fill:currentStatus.iconColor ? currentStatus.iconColor : 'currentColor' }} size={30} />
              )}
            </div>
          )} */}
          <Column>
            {currentStatus.title && (
              <h5 style={isActive ? { color: '#fdd251' } : {}}>{currentStatus.title}</h5>
            )}
            {currentStatus.message && <p>{currentStatus.message}</p>}
          </Column>
        </Row>
      </Column>
    );
  }
);

const Round = memo(({ locked, active, score, reverse }) => (
  <div className={cx(css.round, reverse && css.roundReverse)}>
    <span className={css.roundScore}>{locked ? '?' : isNumber(score) ? score : '--'}</span>
    {active && (
      <svg
        className={css.roundActiveRing}
        width="38"
        height="38"
        viewBox="0 0 32 32"
        fill={reverse ? 'rgba(255, 255, 255, 0.41)' : '#FFCD1A'}
      >
        <path d="M18.2807 0.164618C12.3614 -0.694271 6.45564 1.85703 3.0051 6.66593L4.75435 7.29334C7.78354 3.37916 12.6977 1.28562 17.6667 1.87651L18.2807 0.164618Z" />
        <path d="M23.6894 4.03653L24.3034 2.32465C29.4194 5.42367 32.3576 11.1471 31.9651 17.0531L30.2158 16.4257C30.3649 11.4783 27.9015 6.73853 23.6894 4.03653Z" />
        <path d="M14.3327 30.1234C20.1178 30.8116 25.8286 27.8612 28.5772 22.6353L30.2649 23.2406C27.157 29.3827 20.4465 32.8117 13.7187 31.8353L14.3327 30.1234Z" />
        <path d="M2.07273 13.1289C0.873323 18.9102 3.40679 24.8175 8.31019 27.9632L7.69618 29.6751C1.88178 26.1528 -1.11935 19.2406 0.384979 12.5235L2.07273 13.1289Z" />
      </svg>
    )}
  </div>
));

const MatchHeader = memo(({ video, sets, leftScore, rightScore }) => {
  const hasOpponent = useMemo(() => Boolean(sets && sets[1]), [sets]);
  return (
    <header className={css.header}>
      <div className={css.headerBg} />
      <Team team={(sets[0] || {}).team} left />
      <Score leftScore={leftScore} rightScore={rightScore} hasOpponent={hasOpponent} />
      <Team team={(sets[1] || {}).team} right />
    </header>
  );
});

const Team = memo(({ team, left, right }) => {
  if (!team) {
    team = {
      team_id: 'mysteryy',
      name: '???',
      image_url: unkownOpponentImg
    };
  }

  const isDefaultAvatar =
    (team.image_url || '').includes('default') || team.image_url === unkownOpponentImg;

  return (
    <>
      <div
        className={cx(
          css.teamBackground,
          { [css.pattern]: isDefaultAvatar },
          { [css.patternTwo]: right }
        )}
        style={{
          backgroundImage: !isDefaultAvatar ? `url(${team && team.image_url})` : null,
          left: left ? 0 : '50%',
          right: right ? 0 : '50%'
        }}
      />
      <Column className={css.team}>
        <TeamAvatar {...team} size={75} />
        <h2>{team.name}</h2>
      </Column>
    </>
  );
});

const Score = ({ leftScore, rightScore, hasOpponent }) => {
  return (
    <div className={css.score}>
      <span className={css.animatedScore} key={`score_left_${leftScore}`}>
        {leftScore}
      </span>
      <span>:</span>
      <span className={css.animatedScore} key={`score_right_${rightScore}`}>
        {hasOpponent ? rightScore : '?'}
      </span>
    </div>
  );
};
