import React, { memo, useMemo, useState, useCallback } from 'react';

import MatchController from 'controllers/match';
import useSocketSubscription from 'hooks/useSocketSubscription';
import useSnack from 'hooks/useSnack';

import css from './match-ended-action.module.scss';
import { Button } from 'bebo-ui';

const MatchEndedAction = memo(({ actingTeam, league, track }) => {
  const snack = useSnack();
  const [requesting, setRequesting] = useState(false);

  let [matchInfo] = useSocketSubscription('/match/team', { team_id: actingTeam.team_id }, null, {
    reFetch: true
  });
  let hasMatchInfo = useMemo(() => Boolean(matchInfo), [matchInfo]);
  if (!matchInfo) {
    matchInfo = [];
  }
  let matchesLeft = 5 - matchInfo.length;
  if (matchesLeft < 0) {
    matchesLeft = 0;
  }

  const onNewMatch = useCallback(() => {
    track('new_match');
    setRequesting(true);
    return MatchController.request(league.league_id, actingTeam.team_id)
      .then(response => {
        if (response && response.result && response.result[0]) {
          track('new_match_success');
          return Promise.resolve();
        }
        setRequesting(false);
        console.log('match request response', response);
        return Promise.resolve();
      })
      .catch(err => {
        setRequesting(false);
        track('new_match_fail');
        console.error('match request error', err);
        snack.onError(err);
        return Promise.resolve();
      });
  }, [actingTeam.team_id, league.league_id, snack, track]);

  const renderContent = useMemo(() => {
    if (!hasMatchInfo) {
      return null;
    }
    if (matchesLeft < 1) {
      return <span className={css.matchEndedActionText}>Weekly matches finished!</span>;
    }
    return (
      <>
        <span className={css.matchEndedActionText}>{`${matchesLeft} weekly ${
          matchesLeft === 1 ? 'match' : 'matches'
        } remaining.`}</span>
        <Button
          primary
          onClick={onNewMatch}
          loading={requesting}
          className={css.matchEndedActionBtn}
        >
          NEW MATCH
        </Button>
      </>
    );
  }, [matchesLeft, requesting, hasMatchInfo, onNewMatch]);

  return <div className={css.matchEndedAction}>{renderContent}</div>;
});

export default MatchEndedAction;
