import React, { memo, useMemo } from 'react';
import Linkify from 'react-linkify';
import cx from 'classnames';

import Section from 'components/section';
import { Column, Row, Avatar } from 'bebo-ui';
import { colors } from '../../theme';

import css from './match-teams.module.scss';

const MatchTeams = ({
  sets,
  isOwnActiveMatch,
  userStates,
  actingTeam,
  onSelectUser,
  selectedUser
}) => {
  const teams = useMemo(() => sets.filter(s => s && s.team).map(s => s.team), [sets]);

  return (
    <>
      {teams.map(team => {
        const isOwnTeam =
          ((team && team.team_id) || 1) === ((actingTeam && actingTeam.team_id) || 2);

        return (
          <Section key={team.team_id} className={css.matchTeam} header={team.name}>
            {team.users.map(user => {
              const userState = userStates[user.user_id];
              const isTracking = Boolean(userState && userState.bran);
              const userSet =
                sets &&
                Array.isArray(sets) &&
                sets.find(
                  set =>
                    set.team &&
                    set.team.users &&
                    set.team.users.findIndex(
                      _user => ((_user && _user.user_id) || 1) === (user.user_id || 2)
                    ) > -1
                );

              const isPlaying = userSet && userSet.state === 'started';

              return (
                <MatchTeamItem
                  selected={selectedUser && selectedUser.user_id === user.user_id}
                  isTracking={isTracking}
                  isPlaying={isPlaying}
                  key={user.user_id}
                  user={user}
                  isOwnTeam={isOwnTeam}
                  isOwnActiveMatch={isOwnActiveMatch}
                  onClick={() => onSelectUser(user)}
                  userState={userState}
                />
              );
            })}
          </Section>
        );
      })}
    </>
  );
};

export default MatchTeams;

const MatchTeamItem = memo(
  ({ selected, isPlaying, user, onClick, isTracking, isOwnActiveMatch, isOwnTeam, userState }) => {
    const bio = user.bio ? (
      <Linkify properties={{ target: '_blank' }}> {user.bio} </Linkify>
    ) : (
      "I haven't filled out my profile yet"
    );

    const status = useMemo(() => {
      if (isOwnActiveMatch && isOwnTeam) {
        return isTracking ? (
          <span style={{ color: colors.colorGreen }}>Tracking</span>
        ) : (
          <span style={{ color: colors.colorRed }}>Not Tracking</span>
        );
      }

      return null;
    }, [isOwnActiveMatch, isOwnTeam, isTracking]);

    return (
      <Column className={css.matchTeamItem} onClick={onClick}>
        <Row className={css.info}>
          <Avatar user={user} size={38} className={css.matchTeamUserImage} />
          <Column className={css.matchTeamUserMeta}>
            <span className={css.username}>{user.username}</span>
            {status && <div className={css.status}>{status}</div>}
          </Column>
          <Row />
          {isPlaying && (
            <button className={cx(css.watch, { [css.selected]: selected })}>
              {selected ? 'Watching' : 'Watch'}
              {selected && (
                <>
                  <span>.</span>
                  <span>.</span>
                  <span>.</span>
                </>
              )}
            </button>
          )}
        </Row>
        <Row className={css.bio}>
          <p>{bio}</p>
        </Row>
      </Column>
    );
  }
);
