import React, { useContext, memo, useEffect, useMemo } from 'react';
import cx from 'classnames';

import { Row, Column, IconButton, NavLink } from 'bebo-ui';
import { NavContext } from 'components/navigation';

import css from './nav-bar.module.scss';

import titlelyfy from 'title';
const DEFAULT_WINDOW_TITLE = 'Bebo - High School Esports';

const useCombineStack = stack => {
  return useMemo(
    () => stack.filter(i => i).reduce((prev, current) => ({ ...prev, ...current }), {}),
    [stack]
  );
};

export const NavBarRenderer = memo(({ viewScrollTop, scrollToTop }) => {
  const { toggle, hasLeft, hasRight, navBarStack } = useContext(NavContext);

  const navBarProps = useCombineStack(navBarStack);

  let {
    title,
    description,
    leftIcon,
    rightIcon,
    leftLink,
    leftAction,
    tabs,
    hideTabs,
    hideTabsOnScroll,
    absolute,
    gameLogo
  } = navBarProps;

  if (hideTabsOnScroll === undefined) {
    hideTabsOnScroll = true;
  }
  hideTabsOnScroll = false;

  const scrolledDown = hideTabsOnScroll && viewScrollTop === 'down';

  const canUseHistoryBack = Boolean(
    !leftLink &&
      (!document.referrer || document.referrer.startsWith(window.location.origin)) &&
      window.history.length > 2
  );

  //update the document title based on the title attribute
  useEffect(() => {
    if (document && document.title) {
      let docTitle = DEFAULT_WINDOW_TITLE;
      if (title) {
        docTitle = `${title} - ${docTitle}`;
      }
      docTitle = titlelyfy(docTitle);
      if (docTitle !== document.title) {
        document.title = docTitle;
      }
    }
  }, [title]);

  return (
    <>
      <Row
        className={cx(
          css.container,
          { [css.absolute]: absolute },
          { [css.scrolledDown]: scrolledDown }
        )}
      >
        <Column className={css.btnContainer}>
          {(hasLeft || leftLink) &&
            (leftIcon ? (
              leftIcon
            ) : leftLink ? (
              <IconButton
                color="#fff"
                size={34}
                icon="back"
                style={{ opacity: 1 }}
                link={canUseHistoryBack ? null : leftLink}
                onClick={() => {
                  if (canUseHistoryBack) {
                    window.history.back();
                  }
                  if (leftAction) {
                    leftAction();
                  }
                }}
              />
            ) : (
              <IconButton color="#fff" size={34} icon="menu" onClick={() => toggle(true)} />
            ))}
        </Column>
        <Column
          onClick={scrollToTop}
          className={cx(
            css.headerContainer,
            { [css.hideDesc]: hideTabsOnScroll },
            { [css.show]: scrolledDown }
          )}
        >
          <Row className={css.headerMeta}>
            {gameLogo && <img src={gameLogo} alt="" />}
            <Column>
              <h2>{title}</h2>
              <p>{description}</p>
            </Column>
          </Row>
        </Column>
        <Column className={css.btnContainer}>
          {!hasRight && rightIcon
            ? rightIcon
            : hasRight &&
              (rightIcon ? (
                rightIcon
              ) : (
                <IconButton color="#fff" size={34} icon="menu" onClick={() => toggle(true)} />
              ))}
        </Column>
      </Row>
      {tabs && !hideTabs && (
        <ul className={cx(css.tabs, { [css.hide]: scrolledDown })}>
          {tabs.map(({ link, title, ...rest }, i) => (
            <li className={css.tabItem} key={`tab_${i}`}>
              <NavLink
                className={css.tabLink}
                activeClassName={css.tabLinkActive}
                to={link}
                {...rest}
              >
                {title}
              </NavLink>
            </li>
          ))}
        </ul>
      )}
    </>
  );
});

const NavBar = props => {
  const { addToNavBar } = useContext(NavContext);
  useEffect(() => {
    let removeFromNavBar = addToNavBar(props);
    return () => removeFromNavBar();
  }, [addToNavBar, props]);

  return null;
};

export default memo(NavBar);
