import React, { useState, memo, useCallback, useEffect } from 'react';

import posed from 'react-pose';
import css from './navigation.module.scss';

import { useMedia } from 'the-platform';

// import { TunnelPlaceholder } from 'react-tunnels';

export const NavContext = React.createContext({
  open: () => null,
  close: () => null,
  toggle: () => null,
  hasLeft: false,
  hasRight: false,
  navBarStack: [],
  addToNavBar: () => null
});

const NAV_PERCENT = 0.85;

const Center = posed.div({
  left: {
    x: `${100 * NAV_PERCENT}vw`,

    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  },
  right: {
    x: `${-100 * NAV_PERCENT}vw`,

    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  },
  closed: {
    x: `0vw`,

    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  }
});

const CenterOverlay = posed.div({
  left: {
    applyAtStart: { pointerEvents: 'all' },
    opacity: 0.75,
    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  },
  right: {
    applyAtStart: { pointerEvents: 'all' },
    opacity: 0.9,
    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  },
  closed: {
    applyAtEnd: { pointerEvents: 'none' },
    opacity: 0,
    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  }
});

const SideBar = posed.aside({
  open: {
    applyAtStart: { zIndex: 2, opacity: 1 },
    x: `0%`,
    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  },
  closed: {
    applyAtEnd: { zIndex: 1, opacity: 0 },
    x: ({ isRight }) => (isRight ? `20%` : `-20%`),
    transition: {
      type: 'spring',
      stiffness: 750,
      damping: 50
    }
  }
});

const Navigation = memo(({ left = null, right = null, children }) => {
  //we don't want scrolling on the body
  useEffect(() => {
    const bodyElem = document.getElementsByTagName('body')[0];
    const htmlElem = document.getElementsByTagName('html')[0];
    bodyElem.style.overflow = 'hidden';
    htmlElem.style.overflow = 'hidden';
    return () => {
      bodyElem.style.overflow = null;
      htmlElem.style.overflow = null;
    };
  }, []);

  const [pose, setPose] = useState('closed');
  const [fullyClosed, setFullyClosed] = useState(true);
  const [navBarStack, setNavBarStack] = useState([]);
  const noMobile = useMedia({ minWidth: 1024 }, window.innerWidth > 1024);

  const close = e => {
    if (pose !== 'closed') {
      setPose('closed');
      setFullyClosed(false);
    }
  };

  const open = (left = false) => {
    setFullyClosed(false);
    setPose(left ? 'left' : 'right');
  };

  const toggle = (left = false) => {
    if (pose === 'closed') {
      return open(left);
    } else {
      return close();
    }
  };

  const offsetPercent = (1 - NAV_PERCENT) * 100;

  const leftSideBarPose = pose === 'left' ? 'open' : 'closed';

  const addToNavBar = useCallback(options => {
    setNavBarStack(prevStack => prevStack.concat(options));
    return () => {
      setNavBarStack(prevStack => prevStack.filter(o => o !== options));
    };
  }, []);

  return (
    <NavContext.Provider
      value={{
        open,
        close,
        toggle,
        hasLeft: !noMobile && Boolean(left),
        hasRight: !noMobile && Boolean(right),
        navBarStack,
        addToNavBar
      }}
      className={css.navWrapper}
    >
      {left &&
        (noMobile ? (
          <div className={css.desktopSideBar}>{left}</div>
        ) : (
          <SideBar
            className={css.sidebar}
            style={{ right: `${offsetPercent}vw` }}
            pose={leftSideBarPose}
            hidden={fullyClosed}
          >
            {left}
          </SideBar>
        ))}
      {noMobile ? (
        <div className={css.desktopCenterWrapper}>{children}</div>
      ) : (
        <Center onClick={close} className={css.centerWrapper} pose={pose}>
          <CenterOverlay
            style={{
              background: '#000',
              position: 'absolute',
              top: 0,
              left: 0,
              bottom: 0,
              right: 0,
              zIndex: 100
            }}
          />
          {children}
        </Center>
      )}
    </NavContext.Provider>
  );
});

export default Navigation;
