import React, { memo, useState } from 'react';

import { Column, Button, Input, TeamAvatar, ImageUpload } from 'bebo-ui';

import css from '../settings.module.scss';
import TeamAvatarPlaceholder from '../assets/team.png';

import TeamController from 'controllers/team';

import useTrack from 'hooks/useTrack';
import useSnack from 'hooks/useSnack';

const CreateTeam = memo(({ leagueId, onCreate }) => {
  const [team_name, setTeamName] = useState('');
  const [team_image, setTeamImage] = useState(null);

  const { onError } = useSnack();

  const track = useTrack('create-team', 'league', { league_id: leagueId });
  const signupTrack = useTrack('signup_funnel', 'create-team', { league_id: leagueId });

  const createTeam = () => {
    signupTrack('sumbit', { team_name, team_image });
    TeamController.create(leagueId, team_name, team_image)
      .then(response => {
        console.log('created team', response);
        signupTrack('success', { team_name, team_image });
        if (onCreate) {
          onCreate();
        }
      })
      .catch(err => {
        console.error('failed to create team', err);
        signupTrack('fail', { team_name, team_image });
        onError(err);
      });
  };

  const handleSubmit = () => {
    track('create');
    createTeam();
  };

  return (
    <Column>
      <h3 className={css.settingsHeader}>Create Your Team</h3>

      <ImageUpload
        className={css.avatarContainer}
        size={80}
        onUploadComplete={setTeamImage}
        nameSpace={'team_image'}
        imageComponent={
          <TeamAvatar size={80} image_url={team_image || TeamAvatarPlaceholder} name={team_name} />
        }
      />

      <label className={css.settingsLabel}>Team Name</label>
      <Input
        className={css.settingsInput}
        value={team_name}
        onChange={setTeamName}
        type="text"
        placeholder={'ex. Wizards'}
        fluid
        autoFocus
        onSubmit={handleSubmit}
      />

      <Button disabled={!team_name} primary className={css.settingsButton} onClick={handleSubmit}>
        Join This League
      </Button>
    </Column>
  );
});

export default CreateTeam;
