import React, { memo, useState } from 'react';

import { Column, Button, Input, TeamAvatar, ImageUpload } from 'bebo-ui';

import css from '../settings.module.scss';
import TeamAvatarPlaceholder from '../assets/team.png';

import TeamController from 'controllers/team';

import useTrack from 'hooks/useTrack';
import useSnack from 'hooks/useSnack';

const EditTeam = memo(({ leagueId, onUpdate, team }) => {
  const [team_name, setTeamName] = useState(team.name || '');
  const [team_image, setTeamImage] = useState(team.image_url || '');

  const { onError } = useSnack();

  const track = useTrack('edit-team', 'league', { league_id: leagueId, team_id: team.team_id });

  const editTeam = () => {
    TeamController.update(leagueId, team.team_id, { name: team_name, image_url: team_image })
      .then(response => {
        console.log('updated team', response);

        if (onUpdate) {
          onUpdate();
        }
      })
      .catch(err => {
        console.error('failed to update team', err);
        onError(err);
      });
  };

  const handleSubmit = () => {
    track('edit');
    editTeam();
  };

  return (
    <Column>
      <h3 className={css.settingsHeader}>Edit Team</h3>

      <ImageUpload
        className={css.avatarContainer}
        size={80}
        onUploadComplete={setTeamImage}
        nameSpace={'team_image'}
        imageComponent={
          <TeamAvatar size={80} image_url={team_image || TeamAvatarPlaceholder} name={team_name} />
        }
      />

      <label className={css.settingsLabel}>Team Name</label>
      <Input
        className={css.settingsInput}
        value={team_name}
        onChange={setTeamName}
        type="text"
        placeholder={team.name}
        fluid
        autoFocus
        onSubmit={handleSubmit}
      />

      <Button disabled={!team_name} primary className={css.settingsButton} onClick={handleSubmit}>
        Save Changes
      </Button>
    </Column>
  );
});

export default EditTeam;
