import React, { memo, useState, useEffect } from 'react';
import config from 'scripts/config';

import cx from 'classnames';

import { isDesktopApp } from 'scripts/utils';

import { Column, Button, Input, Loader, Avatar, ImageUpload, Icon, Toggle, Select } from 'bebo-ui';

import useActingUser from 'hooks/useActingUser';

import css from '../settings.module.scss';

import UserController from 'controllers/user';

import useTrack from 'hooks/useTrack';
import { GET, PUT, WS_RAW } from 'classes/socket';

import useUserSettings from 'hooks/useUserSettings';

const NO_WEBCAM = { device_id: 'none', display_name: 'No Camera', type: 'video' };
const NO_MIC = { device_id: 'none', display_name: 'No Microphone', type: 'audio' };

const MIC_DEVICE_KEY = 'mic_device';
const CAM_DEVICE_KEY = 'cam_device';
const DESKTOP_AUDIO_KEY = 'mute_desktop_audio';

const preFillNoDevice = (arr, type) => {
  return [type === 'video' ? NO_WEBCAM : NO_MIC].concat(arr);
};

const getDevices = (retries = 0) => {
  return WS_RAW('/device/list')
    .then(device_response => {
      let video_devices,
        audio_devices = [];
      if (device_response && device_response.video_devices) {
        video_devices = preFillNoDevice(device_response.video_devices, 'video');
      }
      if (device_response && device_response.audio_devices) {
        audio_devices = preFillNoDevice(device_response.audio_devices, 'audio');
      }
      return { video_devices, audio_devices };
    })
    .catch(err => {
      console.error('failed to get mercy devices', err);
      if (retries > 5) {
        return Promise.reject(err);
      }
      return getDevices(retries + 1);
    });
};

const PlayerSettings = memo(({ onClose }) => {
  const actingUser = useActingUser();
  const track = useTrack('settings', 'user');

  const [bio, setBio] = useState((actingUser && actingUser.bio) || '');
  const [username, setUsername] = useState((actingUser && actingUser.username) || '');
  // const [mixername, setMixerName] = useState((actingUser && actingUser.mixer_username) || '');

  const [webcam, setWebcam] = useUserSettings(CAM_DEVICE_KEY);
  const [mic, setMic] = useUserSettings(MIC_DEVICE_KEY);
  const [desktopAudio, setDesktopAudio] = useUserSettings(DESKTOP_AUDIO_KEY);

  const [webcamDevices, setWebcamDevices] = useState([NO_WEBCAM]);
  const [micDevices, setMicDevices] = useState([NO_MIC]);

  const [twitchConnection, setTwitchConnection] = useState(null);

  //get initial devices
  useEffect(() => {
    if (isDesktopApp()) {
      getDevices().then(({ video_devices, audio_devices }) => {
        setWebcamDevices(video_devices);
        setMicDevices(audio_devices);
      });
    }
  }, []);

  const onCamChange = ({ label, value, ...rest }) => {
    const cam = { display_name: label, device_id: value, ...rest };

    if (cam.device_id === 'none') {
      setWebcam(null);
    } else {
      setWebcam(cam.device_id);
    }
  };

  const onMicChange = ({ label, value, ...rest }) => {
    const mic = { display_name: label, device_id: value, ...rest };

    if (mic.device_id === 'none') {
      setMic(null);
    } else {
      setMic(mic.device_id);
    }
  };

  const onMessage = message => {
    if (message && message.data && message.data.type) {
      if (message.data.type === 'oauth-connect') {
        GET('/user/egress')
          .then(({ result }) => {
            const twitch = result.find(item => item.type === 'twitch');
            setTwitchConnection(twitch);
          })
          .catch(err => {
            console.error('failed to GET /user/egress', err);
          });
      } else if (message.data.type === 'retry-auth') {
        window.open(
          `${config.PHARAH_URL}/social/twitch`,
          'twitch_login',
          `width=500,height=${window.innerHeight > 800 ? window.innerHeight : 800}`
        );
      }
    }
  };

  useEffect(() => {
    window.addEventListener('message', onMessage);
    return () => {
      window.removeEventListener('message', onMessage);
    };
  }, []);

  useEffect(() => {
    GET('/user/egress')
      .then(({ result }) => {
        const twitch = result.find(item => item.type === 'twitch');
        setTwitchConnection(twitch);
      })
      .catch(err => {
        console.error('failed to GET /user/egress', err);
      });
  }, []);

  const connectTwitch = () => {
    window.open(
      `${config.PHARAH_URL}/social/twitch`,
      'twitch_login',
      `width=500,height=${window.innerHeight > 800 ? window.innerHeight : 800}`
    );
  };

  const onUserEgressToggle = ({ type, active }) => {
    PUT('/user/egress', { type, active: !active })
      .then(({ result }) => {
        const twitch = result.find(item => item.type === 'twitch');
        setTwitchConnection(twitch);
      })
      .catch(err => {
        console.error('failed to GET /user/egress', err);
      });
  };

  const onUserNameSave = username => {
    return UserController.update({ username });
  };

  const onBioAutoSave = bio => {
    return UserController.update({ bio });
  };

  const onUploadComplete = image_url => {
    return UserController.update({ image_url })

      .then(() => {
        track('change_img');
      })
      .catch(err => {
        console.error('failed to set image_url', err);
      });
  };

  if (!actingUser) {
    return <Loader />;
  }

  return (
    <Column>
      <h3 className={css.settingsHeader}>Player Settings</h3>
      <Button className={css.logoutBtn} onClick={() => UserController.logout()}>
        Logout
      </Button>
      <ImageUpload
        className={css.avatarContainer}
        size={80}
        onUploadComplete={onUploadComplete}
        nameSpace={'profile_image'}
        imageComponent={<Avatar size={80} user={actingUser} />}
      />

      <label className={css.settingsLabel}>Username</label>
      <Input
        className={css.settingsInput}
        value={username}
        onChange={setUsername}
        onAutoSave={onUserNameSave}
        type="text"
        placeholder={'Your Bebo username'}
        fluid
        spellCheck="false"
        autoFocus
      />

      <label className={css.settingsLabel}>Bio</label>
      <Input
        className={css.settingsInput}
        value={bio}
        onChange={setBio}
        onAutoSave={onBioAutoSave}
        type="text"
        placeholder={'Who are you? What team are you on?'}
        fluid
        multiLine={{
          maxRows: 3,
          minRows: 2
        }}
        style={{ height: 'auto' }}
        maxLength={110}
      />

      {isDesktopApp() && (
        <div style={{ display: 'flex', flexDirection: 'row' }}>
          <div style={{ flexGrow: 1, marginRight: 5 }}>
            <label className={css.settingsLabel} style={{ marginBottom: 5, display: 'block' }}>
              Camera
            </label>
            <Select
              value={webcam || 'none'}
              options={webcamDevices.map(({ device_id, display_name, ...rest }) => ({
                value: device_id,
                label: display_name,
                ...rest
              }))}
              onChange={onCamChange}
            />
          </div>
          <div style={{ flexGrow: 1, marginLeft: 5, marginRight: 5 }}>
            <label className={css.settingsLabel} style={{ marginBottom: 5, display: 'block' }}>
              Microphone
            </label>
            <Select
              value={mic || 'none'}
              options={micDevices.map(({ device_id, display_name, ...rest }) => ({
                value: device_id,
                label: display_name,
                ...rest
              }))}
              onChange={onMicChange}
            />
          </div>
          <div style={{ flexGrow: 1, marginLeft: 5 }}>
            <label className={css.settingsLabel} style={{ marginBottom: 5, display: 'block' }}>
              Desktop Audio
            </label>
            <div
              style={{
                background: '#19171d',
                width: '100%',
                borderRadius: 5,
                height: 34,
                display: 'flex'
              }}
            >
              <Toggle
                style={{ margin: 'auto' }}
                onChange={() => {
                  setDesktopAudio(!desktopAudio);
                }}
                value={!desktopAudio}
              />
            </div>
          </div>
        </div>
      )}

      <label className={css.settingsLabel}>Connections</label>
      {twitchConnection ? (
        <div className={cx(css.socialConnection, css.twitch)}>
          <Icon icon="twitch" size={30} />
          <p>Stream to Twitch</p>
          <Toggle
            onChange={() => {
              onUserEgressToggle(twitchConnection);
            }}
            value={twitchConnection.active}
          />
        </div>
      ) : (
        <Button onClick={connectTwitch} className={css.twitchButton} primary>
          <Icon icon="twitch" size={20} />
          Connect Twitch
        </Button>
      )}

      <Button onClick={onClose} className={css.settingsButton} primary>
        Save
      </Button>
    </Column>
  );
});

export default PlayerSettings;
