import React, { memo, useState, useMemo } from 'react';
import cx from 'classnames';

import { Link } from '@reach/router';
import Linkify from 'react-linkify';

import { getParameterByName } from 'scripts/utils';

import { colors } from '../../theme.js';

import css from './sheet-team.module.scss';

import { Avatar, TeamAvatar, Button, ImageUpload, Input, Icon } from 'bebo-ui';
import TeamController from 'controllers/team';

import MatchList from '../match-list';

import useSnack from 'hooks/useSnack';

const SheetTeam = memo(
  ({ team, wins, losses, draws, matchesLeft, matches, qualified, canEdit, ...rest }) => {
    const [activeTab, setActiveTab] = useState('recent_matches');
    const [editing, setEditing] = useState(
      canEdit ? getParameterByName('edit', window.location.toString()) : false
    );
    const [teamName, setTeamName] = useState(team.name);
    const { onError } = useSnack();

    const hasMultipleTabs = useMemo(() => {
      if (team && team.users && team.users.length === 1) {
        return false;
      }
      return true;
    }, [team]);

    const toggleEditing = () => {
      if (canEdit) {
        setEditing(!editing);
      }
    };

    const TABS = useMemo(() => {
      if (!hasMultipleTabs) {
        return [
          {
            id: 'recent_matches',
            label: 'Recent Matches'
          }
        ];
      }
      return [
        {
          id: 'recent_matches',
          label: 'Recent Matches'
        },
        {
          id: 'members',
          label: 'Members'
        }
      ];
    }, [hasMultipleTabs]);

    const tabToRender = useMemo(() => (hasMultipleTabs ? activeTab : 'recent_matches'), [
      hasMultipleTabs,
      activeTab
    ]);

    const onTabChange = tab => {
      if (tab === activeTab) {
        return;
      }
      setActiveTab(tab);
    };

    const onTeamImageChange = image_url => {
      return TeamController.update(team.league_id, team.team_id, { image_url }).catch(err => {
        onError(err);
      });
    };

    const onTeamNameChange = name => {
      return TeamController.update(team.league_id, team.team_id, { name });
    };

    const renderTab = tab => {
      switch (tab) {
        case 'members':
          return <MembersList users={team.users} owner_id={team.owner_id} />;
        case 'recent_matches':
          return <MatchList matches={matches} team_id={team.team_id} league_id={team.league_id} />;
        default:
          break;
      }
    };

    return (
      <div className={css.container}>
        {canEdit && (
          <Button className={css.editBtn} secondary onClick={toggleEditing}>
            {editing ? 'Done' : 'Edit'}
          </Button>
        )}
        <div className={css.header}>
          {editing ? (
            <ImageUpload
              className={css.avatarContainer}
              size={70}
              onUploadComplete={onTeamImageChange}
              nameSpace={'team_image'}
              imageComponent={<TeamAvatar size={70} {...team} />}
            />
          ) : (
            <TeamAvatar {...team} size={70} />
          )}
          {editing ? (
            <Input
              style={{ marginTop: 20 }}
              className={css.teamname}
              value={teamName}
              onChange={setTeamName}
              onAutoSave={onTeamNameChange}
              type="text"
              placeholder="Your Team Name"
              spellCheck="false"
              autoFocus
            />
          ) : (
            <span className={css.teamname}>
              {team.name || (team && team.users && team.users.map(u => u.username).join(' / '))}
            </span>
          )}
          <div className={css.weekStatus}>
            {matchesLeft ? (
              <>
                <b>{matchesLeft}</b> weekly {matchesLeft === 1 ? 'match' : 'matches'} left to play
              </>
            ) : (
              'Finished all weekly matches'
            )}
          </div>
          {qualified && (
            <div className={css.weekStatus} style={{ marginTop: 5 }}>
              <Icon icon="ticket" className={css.qualified} size={20} />
              <b>Qualified</b> for the{' '}
              <Link to={`/app/league/${team.league_id}/tournament`}>tournament</Link>.
            </div>
          )}
        </div>
        <div className={css.stats}>
          <Statsitem color={colors.blueTeam} value={wins} label="Wins" />
          <Statsitem color={colors.redTeam} value={losses} label="Losses" />
          <Statsitem value={draws} label="Draws" />
        </div>
        <TabNav tabs={TABS} activeTab={tabToRender} onChange={onTabChange} />
        <div className={css.body}>{renderTab(tabToRender)}</div>
      </div>
    );
  }
);

const MembersList = memo(({ users, owner_id }) => {
  return (
    <ul className={css.membersList}>
      {users
        .sort(a => {
          if (a.user_id === owner_id) {
            return -1;
          }
          return 0;
        })
        .map(user => (
          <li key={user.user_id} className={css.membersListItem}>
            <div>
              <Avatar size={32} user={user} className={css.membersListItemAvatar} />
              <span className={css.membersListItemName}>{user.username}</span>
              <span className={css.membersListItemScore}>
                {owner_id === user.user_id ? 'Leader' : 'Member'}
              </span>
            </div>
            {user.bio ? (
              <div>
                <span className={css.memberListItemBio}>
                  <Linkify
                    properties={{
                      target: '_blank'
                    }}
                  >
                    {user.bio}
                  </Linkify>
                </span>
              </div>
            ) : (
              <div>
                <span className={css.memberListItemBio}>I haven't filled out my profile yet</span>
              </div>
            )}
          </li>
        ))}
    </ul>
  );
});

const TabNav = memo(({ tabs, activeTab, onChange }) => {
  return (
    <ul className={css.tabNav}>
      {tabs.map(({ label, id }) => (
        <li
          key={id}
          className={cx(css.tabNavItem, { [css.active]: activeTab === id })}
          onClick={() => onChange(id)}
        >
          <span>{label}</span>
        </li>
      ))}
    </ul>
  );
});

const Statsitem = memo(({ label, value, color }) => {
  return (
    <div className={css.stat}>
      <span className={css.statValue}>{value}</span>
      <label style={{ color }} className={css.statLabel}>
        {label}
      </label>
    </div>
  );
});

export default SheetTeam;
