import React, { memo, useState, useEffect } from 'react';
import cx from 'classnames';

import css from './snackbar.module.scss';

import { SheetInner, Sheet } from '../modal-sheet';
import { Button } from 'bebo-ui';

export const SnackbarContext = React.createContext({
  onError: () => null,
  onNext: () => null
});

const DEFAULT_AUTOHIDE = 5000;

let mounted = false;
let QUEUE = [];

const SnackBarContainer = memo(({ children }) => {
  let isSnacking = false;
  let timeout = 0;

  useEffect(() => {
    mounted = true;

    return () => {
      if (timeout) {
        clearTimeout(timeout);
      }
      mounted = false;
    };
  }, [timeout]);
  const [snack, setSnack] = useState({});
  const [open, setOpen] = useState(false);

  const [hidden, setHidden] = useState(true);
  const handlePoseComplete = pose => {
    if (!mounted) {
      return;
    }

    if (pose === 'closed') {
      isSnacking = false;
      setSnack({});
      setHidden(true);
      executeSnack();
    } else {
      setHidden(false);
    }
  };

  const executeSnack = () => {
    if (isSnacking || !QUEUE.length) {
      return;
    }

    isSnacking = true;
    const [snack, ...rest] = QUEUE;
    QUEUE = rest;

    setSnack(snack);
    setOpen(true);
    if (snack.autoHide) {
      timeout = setTimeout(() => {
        isSnacking = false;
        setOpen(false);
        snack.onFinish('autoHide');
      }, snack.autoHide);
    }
  };

  const handleDismiss = e => {
    if (timeout) {
      clearTimeout(timeout);
    }
    isSnacking = false;
    setOpen(false);
  };

  const onNext = ({ action, autoHide, onClick, ...rest }) => {
    return new Promise(resolve => {
      if ((!action && !autoHide) || (autoHide !== false && !autoHide)) {
        autoHide = DEFAULT_AUTOHIDE;
      }
      let onFinish = resolve;
      QUEUE = [
        ...QUEUE,
        {
          action,
          autoHide,
          onClick: e => {
            handleDismiss(e);
            onFinish('click');
            if (onClick) {
              onClick(e);
            }
          },
          ...rest,
          onFinish
        }
      ];

      executeSnack();
    });
  };

  const onError = params => onNext({ ...params, type: 'error' });
  const onSuccess = params => onNext({ ...params, type: 'success' });

  return (
    <SnackbarContext.Provider value={{ onError, onNext, onSuccess }}>
      <Sheet
        className={css.container}
        onPoseComplete={handlePoseComplete}
        pose={open ? 'open' : 'closed'}
        initialPose="closed"
        hidden={!open && hidden}
      >
        <SheetInner className={cx(css.snack, css[snack.type])}>
          <pre>{snack.text}</pre>
          {snack.action && (
            <Button secondary onClick={snack.onClick}>
              {snack.action}
            </Button>
          )}
        </SheetInner>
      </Sheet>
      {children}
    </SnackbarContext.Provider>
  );
});

export default SnackBarContainer;
