import React, { useMemo } from 'react';

import cx from 'classnames';

import useLoadTimes from 'hooks/useLoadTimes';
import useSocketSubscription from 'hooks/useSocketSubscription';
import useEntity from 'hooks/useEntity';

import Countdown, { zeroPad } from 'react-countdown-now';

import { TeamAvatar } from 'bebo-ui';

import css from './storm.module.scss';

function StormTournament({ tournament, league, actingUser, actingTeam }) {
  useLoadTimes();

  const tournament_id = tournament && tournament.tournament_id;

  //   console.log('*** tournament', tournament);

  const tournamentRound = useMemo(() => {
    if (!tournament || !tournament.round) {
      return 0;
    }
    return parseInt(tournament.round);
  }, [tournament]);

  //   const tournamentState = useMemo(() => {
  //     if (!tournament) {
  //       return null;
  //     }
  //     return tournament.state;
  //   }, [tournament]);

  //   const [unfilteredTeams] = useSocketSubscription(
  //     '/tournament/team',
  //     { tournament_id },
  //     undefined,
  //     {
  //       initOptions: { count: 500 }
  //     }
  //   );

  //   const teams = Array.isArray(unfilteredTeams)
  //     ? unfilteredTeams.filter(team => Boolean(team.checkin_dttm))
  //     : unfilteredTeams;

  const [initialRoundMatches] = useSocketSubscription(
    '/tournament/storm',
    {
      tournament_id,
      tournament_round: tournamentRound
    },
    undefined,
    {
      initOptions: { count: 500 },
      pollIf: result => (!result || (Array.isArray(result) && !result.length) ? 20 * 1000 : false)
    }
  );

  const entityMatches = useEntity(
    'match',
    (initialRoundMatches || []).map(match => match.match_id)
  );

  const sortedMatches = entityMatches
    .sort((a, b) => b.tournament_order - a.tournament_order)
    .map((m, i) => ({ ...m, _ind: i }));

  const winMatches = sortedMatches.filter(match => match.tournament_bracket === 'win');
  const aliveMatches = sortedMatches.filter(match => match.tournament_bracket === 'alive');
  const stormMatches = sortedMatches.filter(match => match.tournament_bracket === 'storm');
  const eliminatedMatches = sortedMatches.filter(
    match => match.tournament_bracket === 'eliminated'
  );

  return (
    <div className={css.container}>
      <StormCountDown
        nextStorm={tournament.next_round_dttm}
        isEnded={Boolean(tournament && tournament.end_dttm)}
      />
      <StormBracket matches={winMatches} type="winner" />
      <StormBracket matches={aliveMatches} type="alive" />
      <StormBracket matches={stormMatches} type="storm" />
      <StormBracket matches={eliminatedMatches} type="eliminated" />
    </div>
  );
}

export default StormTournament;

const StormCountDown = ({ nextStorm, isEnded }) => {
  if (isEnded) {
    return (
      <div className={css.timer}>
        <h2>This tournament is over.</h2>
      </div>
    );
  }
  if (!nextStorm) {
    return null;
  }
  return (
    <Countdown
      date={nextStorm}
      renderer={({ days, hours, minutes, seconds, completed }) => {
        if (completed) {
          return null;
        }
        return (
          <div className={css.timer}>
            <h2>
              Time to next storm:{' '}
              <span>
                {days ? `${zeroPad(days)}:` : ''}
                {hours ? `${zeroPad(hours)}:` : ''}
                {zeroPad(minutes)}:{zeroPad(seconds)}
              </span>
            </h2>
          </div>
        );
      }}
    />
  );
};

const StormBracket = ({ matches, type }) =>
  matches.length ? (
    <section className={css[`bracket-${type}`]}>
      <div className={css.wrapper}>
        <div className={css.bracketTitle}>
          <p>{type}</p>
        </div>
        {matches.map(match => (
          <StormMatchCard key={match.match_id} match={match} type={type} />
        ))}
      </div>
    </section>
  ) : null;

const StormMatchCard = ({ match, type }) => {
  const set = match.sets[0];

  return (
    <div className={cx(css.matchCardTeam, match._ind % 2 === 0 ? css.even : null)}>
      <TeamAvatar className={css.matchTeamAvatar} {...set.team} size={40} />
      <div className={css.matchCardTeamMeta}>
        <h4>
          {(set && set.team && set.team.name) ||
            (set && set.team && set.team.users && set.team.users.map(u => u.username).join(' / '))}
          <br />
          <span>
            ({set && set.team && set.team.users && set.team.users.map(u => u.username).join(' / ')})
          </span>
        </h4>
        <h4 className={css[type]}>{type}</h4>
      </div>
      <h3>{set.scores.reduce((a, b) => a + b, 0)}</h3>
    </div>
  );
};
