import React, { memo, useEffect, useMemo, useState } from 'react';

import { Icon, Row, Button, Column } from 'bebo-ui';
import cx from 'classnames';
import imageArt from './assets/tournament-art.png';

import { format, subHours, isAfter } from 'date-fns';

import css from './tournament.module.scss';

import titlelyfy from 'title';

import Countdown, { zeroPad } from 'react-countdown-now';

import { GET, POST, DELETE } from 'classes/socket';
import { getSingleResult } from 'scripts/utils';

// const SIZES = ['small', 'medium', 'large'];

const POLL_INTERVAL = 30 * 1000; //30s

const UpcomingTournament = memo(({ tournament, actingTeam, league }) => {
  const { name, prize, description, start_dttm, points_needed } = tournament;
  const { points } = actingTeam;

  const [tournamentTeams, setTournamentTeams] = useState([]);
  const [pollInterval, setPollInterval] = useState(0);

  const checkin_dttm = useMemo(() => subHours(start_dttm, 1), [start_dttm]);

  useEffect(() => {
    let t = setTimeout(() => {
      setPollInterval(pollInterval + 1);
    }, POLL_INTERVAL);
    if (isAfter(new Date(), checkin_dttm)) {
      GET('/tournament/team', { count: 500, tournament_id: tournament.tournament_id })
        .then(response => {
          if (response && response.result) {
            setTournamentTeams(response.result);
          }
        })
        .catch(err => {
          console.error('failed to get tournament /team', err);
        })
        .finally(() => {
          if (t) {
            clearTimeout(t);
          }
          t = setTimeout(() => {
            setPollInterval(pollInterval + 1);
          }, POLL_INTERVAL);
        });
    }
    return () => {
      if (t) {
        clearTimeout(t);
      }
    };
  }, [checkin_dttm, pollInterval, tournament.tournament_id]);

  // const startsWithin24h = useMemo(() => {
  //   return differenceInHours(start_dttm, new Date()) < 24;
  // }, [start_dttm]);

  const myTournamentTeam = useMemo(() => {
    return tournamentTeams.find(
      t => ((t && t.team_id) || 1) === ((actingTeam && actingTeam.team_id) || 2)
    );
  }, [tournamentTeams, actingTeam]);

  const isCheckedIn = Boolean(myTournamentTeam && myTournamentTeam.checkin_dttm);

  const doCheckIn = () => {
    if (!myTournamentTeam || !myTournamentTeam.tournament_team_id) {
      return Promise.reject('not on a tournament_team');
    }
    const tournament_team_id = myTournamentTeam.tournament_team_id;
    return POST('/tournament/team/checkin', {
      tournament_team_id
    })
      .then(response => {
        const checkedInTeam = getSingleResult(response);
        let prevTournamentTeams = tournamentTeams.slice();
        let index = prevTournamentTeams.findIndex(t => t.tournament_team_id === tournament_team_id);
        const myNewTournamentTeam = { ...myTournamentTeam, ...checkedInTeam };
        prevTournamentTeams[index] = myNewTournamentTeam;
        setTournamentTeams(prevTournamentTeams);
      })
      .catch(err => {
        console.error('failed to do tournament checkin', err);
      });
  };

  const doCheckOut = () => {
    if (!myTournamentTeam || !myTournamentTeam.tournament_team_id) {
      return Promise.reject('not on a tournament_team');
    }
    const tournament_team_id = myTournamentTeam.tournament_team_id;
    return DELETE('/tournament/team/checkin', {
      tournament_team_id
    })
      .then(response => {
        const checkedInTeam = getSingleResult(response);
        let prevTournamentTeams = tournamentTeams.slice();
        let index = prevTournamentTeams.findIndex(t => t.tournament_team_id === tournament_team_id);
        const myNewTournamentTeam = { ...myTournamentTeam, ...checkedInTeam };
        prevTournamentTeams[index] = myNewTournamentTeam;
        setTournamentTeams(prevTournamentTeams);
      })
      .catch(err => {
        console.error('failed to do tournament checkout', err);
      });
  };

  return (
    <div className={css.page}>
      <div className={css.header}>
        <span className={css.headerName}>{name}</span>
        <span className={css.headerDate}>{format(start_dttm, 'MMM do, p')}</span>
        <HeaderBackground imageArt={imageArt} leagueImage={league.image_url} />
      </div>
      <div className={css.body}>
        <TeamStatus
          checkin_dttm={checkin_dttm}
          points_needed={parseInt(points_needed, 10)}
          points={points}
          isCheckedIn={isCheckedIn}
          doCheckIn={doCheckIn}
          doCheckOut={doCheckOut}
        />
        {prize && <Section heading={'Prize'} text={prize} />}
        {description && <Section heading={'Description'} text={description} />}
      </div>
    </div>
  );
});

export default UpcomingTournament;

const Section = memo(({ heading, text }) => {
  return (
    <div className={css.section}>
      <h3>{heading}</h3>
      <p>{text}</p>
      <div />
    </div>
  );
});

const HeaderBackground = memo(({ imageArt, leagueImage }) => {
  return (
    <div className={css.headerBackground}>
      <div className={css.headerBackgroundGradient} />
      <div className={css.headerBackgroundLeague}>
        <div />
        <img alt="" src={leagueImage} />
      </div>
      <img alt="" src={imageArt} className={css.headerBackgroundArt} />
    </div>
  );
});

const getStatus = (checkedIn, isCheckinTime, points_needed, points) => {
  if (isCheckinTime) {
    if (checkedIn) {
      return {
        icon: 'checkmark',
        header: 'You are checked in!',
        desc: 'HF & GL in the tournament!',
        btn: `Can't play? Undo check-in.`
      };
    }
    return {
      icon: 'close',
      header: 'Not yet checked in!',
      desc: 'You are eligible but you still need to check in to participate in the tournament!',
      btn: `Check in now!`
    };
  } else {
    return {
      icon: 'ticket',
      header: 'You are eligible!',
      desc: 'Make sure you show up on time to check in! (Starts 1h before the tournament.)'
    };
  }
};

const TeamStatus = memo(
  ({ points_needed, points, checkin_dttm, isCheckedIn, doCheckIn, doCheckOut }) => {
    const [loading, setLoading] = useState(false);

    const handleActionClick = () => {
      setLoading(true);
      let prom = Promise.resolve();
      if (isCheckedIn) {
        prom = doCheckOut();
      } else {
        prom = doCheckIn();
      }

      prom.finally(() => {
        setLoading(false);
      });
    };

    return (
      <Countdown
        date={checkin_dttm}
        renderer={({ days, hours, minutes, seconds, completed }) => {
          const status = getStatus(isCheckedIn, completed, points_needed, points);
          return (
            <Column className={cx(css.teamStatus, { [css.qualified]: isCheckedIn })}>
              <Row>
                <div className={css.teamStatusImage}>
                  <Icon icon={status.icon} className={css.teamStatusImageIcon} />
                </div>
                <span className={css.teamText}>
                  <span className={css.teamTextHeading}>{titlelyfy(status.header)}</span>
                  <span className={css.teamTextDesc}>{status.desc}</span>
                </span>
              </Row>
              <Row className={css.teamStatusActionContainer}>
                <Button
                  loading={loading}
                  onClick={handleActionClick}
                  disabled={loading || !completed}
                  fluid
                  primary
                  inverse={isCheckedIn}
                >
                  {status.btn
                    ? status.btn
                    : `Check in starts: ${days ? `${zeroPad(days)}:` : ''}${zeroPad(
                        hours
                      )}:${zeroPad(minutes)}:${zeroPad(seconds)}`}
                </Button>
              </Row>
            </Column>
          );
        }}
      />
    );
  }
);
