import React, { useMemo, useRef, useState, useLayoutEffect } from 'react';

import useComponentSize from '@rehooks/component-size';

const generateBarBackground = (isActive, isLeft, isRight) => {
  if (isActive) {
    if (isLeft) {
      return `linear-gradient(to left, rgb(0, 132, 255), rgba(0, 132, 255, 0))`;
    } else if (isRight) {
      return `linear-gradient(to right, rgb(0, 132, 255), rgba(0, 132, 255, 0))`;
    }
    return `rgb(0, 132, 255)`;
  }
  if (isLeft) {
    return `linear-gradient(to left, rgb(156, 158, 160), rgb(156, 158, 160, 0))`;
  } else if (isRight) {
    return `linear-gradient(to right, rgb(156, 158, 160), rgb(156, 158, 160, 0))`;
  }
  return `rgb(156, 158, 160)`;
};

const itemWidth = 80;

const ProgressBar = ({ currentRound, rounds, selectedRound, setUserSelectedRound }) => {
  const [hasRendered, setHasRendered] = useState(false);
  useLayoutEffect(() => {
    let t = setTimeout(() => setHasRendered(true), 500);
    return () => clearTimeout(t);
  }, []);
  const rowRef = useRef(null);
  const { width } = useComponentSize(rowRef);

  const translateX = useMemo(() => {
    const itemsThatFit = width / itemWidth;

    const caclulatedCenterItem = selectedRound - itemsThatFit / 2;

    let translateItem = Math.min(Math.max(0, caclulatedCenterItem), rounds.length - itemsThatFit);

    let translateValue = 0;
    if (translateItem <= 0) {
      translateValue = -20;
    } else if (Math.floor(caclulatedCenterItem) > Math.floor(translateItem)) {
      translateValue = itemWidth * translateItem - itemWidth / 2 + 40;
    } else {
      translateValue = Math.max(-20, itemWidth * translateItem - itemWidth / 2);
    }

    return -1 * translateValue;
  }, [rounds, selectedRound, width]);

  return (
    <div style={{ width: '100%', height: '100%', display: 'flex' }} ref={rowRef}>
      <div
        style={{
          display: 'flex',
          flexDirection: 'row',
          transform: `translateX(${translateX}px)`,
          transition: hasRendered ? 'transform 300ms ease' : '',
          minWidth: 'calc(100% - 40px)'
        }}
      >
        {rounds.map((round, i) => (
          <React.Fragment key={`round_${round.index}`}>
            {i !== 0 && (
              <hr
                style={{
                  flexGrow: 1,
                  flexShrink: 0,
                  margin: '11px 0',
                  boxShadow: i < currentRound ? `0px 1px 19px rgba(49, 151, 245, 0.62)` : null,
                  background: generateBarBackground(i < currentRound, i === 0, false),
                  border: 'none',
                  opacity: round.index - 1 === 0 ? 0 : 1,
                  width: 20
                }}
              />
            )}

            <div
              onClick={
                round.index <= currentRound && round.index !== selectedRound
                  ? () => setUserSelectedRound(round.index === currentRound ? null : round.index)
                  : null
              }
              style={{
                flexShrink: 0,
                flexGrow: 0,

                border: '1px solid #9C9EA0',
                borderColor:
                  round.index === selectedRound
                    ? '#FBBF25'
                    : round.index <= currentRound
                    ? '#0084FF'
                    : '#9C9EA0',
                borderRadius: '50px',
                width: 60,
                height: 23,
                display: 'flex',
                background:
                  round.index <= currentRound
                    ? '#0084FF'
                    : round.isFinal
                    ? 'rgba(149, 146, 153, 0.1)'
                    : null,
                boxShadow:
                  round.index === selectedRound
                    ? '0px 1px 25px rgba(251, 191, 37, 0.62)'
                    : round.index <= currentRound
                    ? `0px 1px 19px rgba(49, 151, 245, 0.62)`
                    : null
              }}
            >
              <span style={{ paddingLeft: 1, margin: 'auto', lineHeight: '20px', fontSize: 11 }}>
                {round.isFinal ? 'Final' : `Round ${round.index}`}
              </span>
            </div>
          </React.Fragment>
        ))}
      </div>
    </div>
  );
};

export default ProgressBar;
