import { getSingleResult } from 'scripts/utils';

import { BehaviorSubject } from 'rxjs';
import { map, distinctUntilChanged, filter } from 'rxjs/operators';

import { GET, PUT, onmessage$ } from 'classes/socket';

import invariant from 'invariant';

const VIPER_PREFIX = 'VIPER_';

const _currentUserSettings = new BehaviorSubject({});

let _onMessageSub = null;
let _fetching = false;

export const init = () => {
  _onMessageSub = onmessage$
    .pipe(filter(m => m && m.url && m.url === '/user/settings'))
    .subscribe(onMessage);
  _fetching = true;
  return GET('/user/settings')
    .then(response => {
      if (_fetching) {
        onMessage(response);
      }
      return Promise.resolve();
    })
    .catch(err => {
      console.error('failed to init user/settings', err);
    })
    .finally(() => {
      _fetching = false;
    });
};
export const cleanup = () => {
  _fetching = false;

  if (_onMessageSub) {
    _onMessageSub.unsubscribe();
    _onMessageSub = null;
  }
  _currentUserSettings.next({});
  return Promise.resolve();
};

const onMessage = message => {
  const viperObj = convertMessage(message);
  _currentUserSettings.next(viperObj);
};

const convertMessage = result => {
  const obj = getSingleResult(result);
  const viper_obj = {};
  Object.keys(obj).forEach(key => {
    if (key.startsWith(VIPER_PREFIX)) {
      viper_obj[key] = obj[key];
    }
  });
  return viper_obj;
};

export const subscribeKey = (key = '') => {
  invariant(key, 'subscribeKey requires a string "key" to be passed to it');
  invariant(!key.startsWith(VIPER_PREFIX), 'subscribeKey key should not begin with "VIPER_"');
  const viperKey = `${VIPER_PREFIX}${key}`;
  return _currentUserSettings.pipe(
    map(obj => obj[viperKey] || null),
    distinctUntilChanged()
  );
};

export const updateKey = (key, value) => {
  return PUT('/user/settings', { [`${VIPER_PREFIX}${key}`]: value })
    .then(onMessage)
    .catch(err => {
      console.error('failed to update /user/settings key', err);
    });
};
