import { GET, POST, PUT } from 'classes/socket';
import { BehaviorSubject } from 'rxjs';
import { distinctUntilChanged } from 'rxjs/operators';
import { getSingleResult, sessionStorage, isDesktopApp, storage } from 'scripts/utils';
import { getUtmValues } from 'scripts/analytics';
import * as Sentry from '@sentry/browser';
import Config from 'scripts/config';
import { init, cleanup } from './user-settings.js';
import Cookies from 'js-cookie';

const cookieName = name => {
  return name + '_' + Config.COOKIE_ENV;
};

const setCookie = (cname, cvalue, exdays) => {
  if (!exdays) {
    exdays = 365; // 1year
  }
  let expires = new Date();
  expires.setDate(expires.getDate() + exdays);

  console.log(
    'setting cookie',
    { cname, cvalue, exdays },
    Config.COOKIE_ENV,
    Config.COOKIE_DOMAIN,
    cookieName(cname)
  );

  Cookies.set(cookieName(cname), cvalue, {
    expires: expires,
    path: '/',
    domain: Config.COOKIE_DOMAIN
  });
};

const removeCookie = name => {
  let expires = new Date();
  expires.setDate(expires.getDate() - 1);

  console.log(
    'removing cookie',
    { name },
    Config.COOKIE_ENV,
    Config.COOKIE_DOMAIN,
    cookieName(name)
  );

  Cookies.set(cookieName(name), '', {
    expires: expires,
    path: '/',
    domain: Config.COOKIE_DOMAIN
  });
};

class UserController {
  getAccessToken = () => {
    if (!isDesktopApp()) {
      return Promise.resolve(sessionStorage.get('at'));
    }
    return Promise.resolve();
  };

  setAccessToken = at => {
    sessionStorage.set('at', at);
    accessToken$.next(at);
  };

  clearAccessToken = () => {
    sessionStorage.set('at', null);
    accessToken$.next(null);
  };

  getUserId = () => {
    const user = loggedInUser$.getValue();
    if (!user) {
      return null;
    }
    return user.user_id;
  };

  checkStatus = () => {
    return this.getAccessToken()
      .then(access_token => {
        if (!access_token && isDesktopApp()) {
          return GET('/user/access_token', null)
            .then(response => {
              if (response.access_token) {
                this.setAccessToken(response.access_token);
                return response.access_token;
              }
              return Promise.resolve();
            })
            .catch(err => {
              console.log('failed to GET /user/access_token', err);
              return Promise.resolve();
            });
        }
        return access_token;
      })
      .then(access_token => {
        if (!access_token) {
          isLoggedIn$.next(false);
          loggedInUser$.next(null);
          return Promise.reject({ code: 401 });
        }

        return GET('/user/me', null)
          .then(response => {
            response = getSingleResult(response);
            console.log('checkStatus', response);
            isLoggedIn$.next(!!(response && response.username));
            loggedInUser$.next(response);
            return init().finally(() => {
              return response;
            });
          })
          .catch(err => {
            console.error('checkStatus', err);
            isLoggedIn$.next(false);
            loggedInUser$.next(null);
            throw err;
          });
      });
  };

  signup = (email, username, password, phone_number, pin) => {
    const code = storage.get('invite-code');
    const payload = { email, username, password, phone_number, pin, ...getUtmValues() };
    if (code) {
      payload.code = code;
    }
    return POST('/user/signup', payload).then(response => {
      const status_code = response.code;
      response = getSingleResult(response);
      response.status_code = status_code;
      console.log('/user/signup response', response);
      this.setAccessToken(response.access_token);
      isLoggedIn$.next(!!(response && response.username));
      loggedInUser$.next(response);
      if (status_code === 200) {
        return init().finally(() => {
          return response;
        });
      }
      return response;
    });
  };

  login = (username, password, phone_number, pin) => {
    return POST('/user/login', { username, password, phone_number, pin }).then(response => {
      const status_code = response.code;
      response = getSingleResult(response);
      response.status_code = status_code;
      console.log('/user/login response', response);
      this.setAccessToken(response.access_token);
      isLoggedIn$.next(!!(response && response.username));
      loggedInUser$.next(response);
      if (status_code === 200) {
        return init().finally(() => {
          return response;
        });
      }
      return response;
    });
  };

  update = update => {
    return PUT('/user', update).then(response => {
      const status_code = response.code;
      response = getSingleResult(response);
      response.status_code = status_code;
      console.log('PUT /user response', response);
      return response;
    });
  };

  logout = () => {
    if (isDesktopApp()) {
      isLoggedIn$.next(false);
      loggedInUser$.next(null);
      sessionStorage.clear();

      return POST('/user/logout')
        .then(response => {
          response = getSingleResult(response);
          this.clearAccessToken();
          return response;
        })
        .catch(err => {
          console.error('failed to log out', err);
          return Promise.resolve();
        });
    }
    return new Promise(resolve => {
      isLoggedIn$.next(false);
      loggedInUser$.next(null);
      sessionStorage.clear();
      this.clearAccessToken();
      return cleanup().then(resolve);
    });
  };
}

const singleton = new UserController();

export default singleton;

export const loggedInUser$ = new BehaviorSubject(null).pipe(
  distinctUntilChanged((a, b) => {
    const compareA = a && a.user_id;
    const compareB = b && b.user_id;
    return compareA === compareB;
  })
);

export const isLoggedIn$ = new BehaviorSubject(null).pipe(distinctUntilChanged());
export const accessToken$ = new BehaviorSubject(singleton.getAccessToken()).pipe(
  distinctUntilChanged()
);

accessToken$.subscribe(at => {
  let atProm = Promise.resolve(at);
  if (at && typeof at !== 'string') {
    atProm = at.then(a => {
      return a;
    });
  }
  atProm.then(resolvedAt => {
    console.log('resolvedAt', resolvedAt);
    if (resolvedAt) {
      setCookie('b_at', resolvedAt);
    } else {
      removeCookie('b_at');
    }
  });
});

loggedInUser$.subscribe(user => {
  if (user && user.user_id) {
    Sentry.configureScope(scope => {
      scope.setUser({ user_id: user.user_id, username: user.username, days_cnt: user.days });
    });
  } else {
    Sentry.configureScope(scope => {
      scope.setUser({ user_id: null, username: null, days_cnt: null });
    });
  }
});
