import { useState, useEffect, useCallback } from 'react';
import { getSubject, removeSubject } from 'subjects/manager';
import { tap, filter } from 'rxjs/operators';
import { getRoutingKey } from 'scripts/utils';

const useSocketSubscription = (
  url,
  filters,
  initialValue,
  {
    isSingleValue,
    noInit = false,
    debug = false,
    reFetch = false,
    initOptions = {},
    pollInterval = false,
    pollIf
  } = {}
) => {
  if (!filters) {
    filters = {};
  }
  if (!isSingleValue) {
    isSingleValue = false;
  }

  const [value, setValue] = useState(initialValue);
  const [hasInitial, setHasInitial] = useState(false);
  const [error, setError] = useState(null);
  const [intervalIteration, setIntervalIteration] = useState(0);

  if (pollIf && typeof pollIf === 'function') {
    let pollIfInterval = pollIf(value);
    if (pollIfInterval && !pollInterval) {
      pollInterval = pollIfInterval;
    }
  }

  let setActualValue = useCallback(
    val => {
      setValue(val);
      if (!hasInitial) {
        setHasInitial(!hasInitial);
      }
    },
    [hasInitial]
  );

  useEffect(() => {
    let timeout = null;
    if (pollInterval) {
      timeout = setTimeout(() => setIntervalIteration(intervalIteration + 1), pollInterval);
    }
    const s = getSubject(
      url,
      filters,
      initialValue,
      isSingleValue,
      noInit,
      reFetch || pollInterval,
      initOptions
    )
      .pipe(
        filter(m => m),
        filter(m => m !== initialValue),
        tap(m => (debug ? console.warn(m) : null))
      )
      .subscribe(setActualValue, setError);

    return () => {
      if (timeout) {
        clearTimeout(timeout);
      }
      s.unsubscribe();
      removeSubject(url, filters, isSingleValue);
    };
    //eslint-disable-next-line
  }, [getRoutingKey(url, filters, isSingleValue), intervalIteration, pollInterval]);

  const clearError = () => setError(null);
  const refresh = useCallback(() => {
    if (hasInitial) {
      setIntervalIteration(intervalIteration + 1);
    }
  }, [intervalIteration, hasInitial]);

  return [value, error, { clearError, refresh }];
};

export default useSocketSubscription;
