import React, { memo, useMemo, useState, useEffect } from 'react';

import { Router } from '@reach/router';

import { FullSizeLoader, Icon } from 'bebo-ui';
import Sheet from 'components/modal-sheet';

import NavBar from 'components/nav-bar';

import css from './league-page.module.scss';

//sub-pages
import Standings from './pages/standings';
import Tournament from './pages/tournament';
import MyTeam from './pages/team';
import Match from './pages/match';

//end sub-pages

import useActingUser from 'hooks/useActingUser';
import useSubscribeTo from 'hooks/useSubscribeTo';
import useEntity from 'hooks/useEntity';
import useSocketSubscription from 'hooks/useSocketSubscription';
import useUserSettings from 'hooks/useUserSettings';

import { POST } from 'classes/socket';
import CopyButton from 'components/copy-button';

import FortniteMark from 'assets/fortnite-mark.png';
import ApexMark from 'assets/apex-white-nav-logo.svg';

//invite campaign stuff

const CAMPAIGN = {
  campaign_name: 'user_viral',
  campaign_source: 'league',
  campaign_medium: 'team_summary_card'
};

//end invite campaign stuff

const STANDINGS_POLL_TIME = 30 * 1000; //30s

function LeaguePage({ league_id }) {
  const [sheet, setSheet] = useState('');
  //do data fetching etc for main league page
  const actingUser = useActingUser();
  useSubscribeTo('/league', { league_id });

  const league = useEntity('league', league_id);

  const gameLogo = useMemo(() => {
    if (!league || !league.game_id) {
      return null;
    }
    if (league.game_id === 'fortnite') {
      return FortniteMark;
    }
    return ApexMark;
  }, [league]);

  const [previousLeagueId, setPreviousLeagueId] = useUserSettings('last-league');
  useEffect(() => {
    if (league_id && previousLeagueId !== league_id) {
      setPreviousLeagueId(league_id);
    }
    //eslint-disable-next-line
  }, [league_id, setPreviousLeagueId]);

  const division_id = league && league.active_division_id;

  const [standings] = useSocketSubscription('/division/standing', { division_id }, [], {
    reFetch: true,
    pollInterval: STANDINGS_POLL_TIME
  });

  const actingTeam = useMemo(() => {
    if (actingUser && league_id) {
      const actingTeam =
        (actingUser.leagues &&
          ((actingUser.leagues.find(l => l.league_id === league_id) || {}).teams || [])[0]) ||
        {};
      if (actingTeam && standings) {
        for (let s of standings) {
          if (((s && s.team && s.team.team_id) || 1) === (actingTeam.team_id || 2)) {
            actingTeam.qualified = s.qualified;
            actingTeam.wins = s.win;
            actingTeam.points = s.points || 0;
            break;
          }
        }
      }
      return actingTeam;
    }
    return {};
  }, [actingUser, league_id, standings]);

  const tabs = [
    { title: 'Home', link: `./league/${league_id}`, exact: true },
    { title: 'Tournaments', link: `./league/${league_id}/tournament` }
  ];
  if (actingTeam.team_id) {
    tabs.push({ title: 'My Team', link: `./league/${league_id}/my-team` });
  }

  const [leagueInviteLink, setLeagueInviteLink] = useUserSettings(`invite_link_${league_id}`);

  const renderLoader = useMemo(() => {
    return !league || league.isWaiting;
  }, [league]);

  const onInvite = () => {
    let prom = Promise.resolve(leagueInviteLink);
    if (!leagueInviteLink) {
      const payload = { ...CAMPAIGN, league_id, viral_user_id: actingUser.user_id };
      prom = POST('/invite', payload).then(response => {
        if (response && response.result && response.result[0]) {
          return response && response.result[0].invite_url;
        }
        return Promise.reject(new Error('failed to get invite url'));
      });
    }
    prom
      .then(invite_url => {
        setLeagueInviteLink(invite_url);
        setSheet('invite');
      })
      .catch(err => {
        console.error('failed to generate invite link', err);
        setSheet('');
      });
  };

  return renderLoader ? (
    <FullSizeLoader />
  ) : (
    <>
      <NavBar
        gameLogo={gameLogo}
        title={league.name}
        tabs={tabs}
        hideTabsOnScroll
        rightIcon={
          <Icon color="#fff" icon="community" onClick={onInvite} style={{ marginRight: 5 }} />
        }
      />
      <Router style={{ width: '100%', height: '100%' }}>
        <Standings
          default
          path="standings"
          actingUser={actingUser}
          standings={standings}
          league={league}
          actingTeam={actingTeam}
        />
        <Tournament
          path="tournament/*"
          actingUser={actingUser}
          league={league}
          actingTeam={actingTeam}
        />
        <MyTeam
          path="my-team"
          actingUser={actingUser}
          league_id={league_id}
          team={actingTeam}
          standings={standings}
        />
        <Match path="match/:match_id" league={league} />
      </Router>

      <Sheet open={sheet === 'invite' && leagueInviteLink} onClose={() => setSheet('')}>
        <InviteSheet inviteUrl={leagueInviteLink} onClose={() => setSheet('')} />
      </Sheet>
    </>
  );
}

export default memo(LeaguePage);

const InviteSheet = memo(({ inviteUrl, onClose }) => {
  return (
    <div className={css.inviteSheet}>
      <h4>Invite Your Friends</h4>
      <p>Know someone who you'd love to play against?</p>
      <p>Send them this link so they can join the league!</p>
      <p>
        <b>{inviteUrl}</b>
      </p>
      <CopyButton
        successText={'Copied invite link successfully.'}
        primary
        fluid
        onCopy={() => {
          onClose();
        }}
        value={inviteUrl}
      >
        Copy Invite Link
      </CopyButton>
    </div>
  );
});
