import React, { memo, useMemo, useEffect, useState } from 'react';

import { Router } from '@reach/router';

import useSubscribeTo from 'hooks/useSubscribeTo';
import useEntity from 'hooks/useEntity';

import NavBar from 'components/nav-bar';

import { Column } from 'bebo-ui';

import StormTournament from 'components/storm-tournament';
import BracketTournament from 'routes/tournament';
import UpcomingTournament from 'components/tournament-v2';
import TournamentListItem from 'components/tournament-list-item';

import { isAfter } from 'date-fns';

import { GET } from 'classes/socket';

import NoTournamentImg from './assets/no_tournaments.png';

const NoTournament = memo(() => (
  <div style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, display: 'flex' }}>
    <img
      style={{ margin: 'auto', maxWidth: '50%', transform: 'translateY(-15%)' }}
      src={NoTournamentImg}
      alt="Currently No Tournament"
    />
  </div>
));

function SingleTournament({ tournament_id, league, actingTeam, actingUser }) {
  const [tournamentError] = useSubscribeTo('/tournament', { tournament_id });
  const tournament = useEntity('tournament', tournament_id);
  if (!tournament || tournamentError) {
    return null;
  }

  const isStormTournament = tournament.type === 'storm';
  const isSingleElim = tournament.type === 'signle_elim';

  return (
    <>
      <NavBar
        sticky
        // absolute
        leftLink={`/app/league/${league.league_id}/tournament`}
        description={tournament.name}
        hideTabs
      />
      {isAfter(new Date(), tournament.start_dttm) ? (
        isStormTournament ? (
          <StormTournament
            tournament={tournament}
            league={league}
            actingUser={actingUser}
            actingTeam={actingTeam}
          />
        ) : (
          <BracketTournament
            singleElim={isSingleElim}
            tournament={tournament}
            league={league}
            actingUser={actingUser}
            actingTeam={actingTeam}
          />
        )
      ) : (
        <UpcomingTournament
          default
          tournament={tournament}
          actingTeam={actingTeam}
          league={league}
        />
      )}
    </>
  );
}

function TournamentList({ league }) {
  const [tournaments, setTournaments] = useState([]);
  const [hasTournaments, setHasTournaments] = useState(false);
  useEffect(() => {
    GET('/tournament', {
      league_id: league.league_id,
      count: 500,
      order: JSON.stringify([['start_dttm', 'DESC']])
    })
      .then(({ result }) => {
        setTournaments(result);
        setHasTournaments(true);
      })
      .catch(err => {
        setHasTournaments(true);
        console.error('failed to GET tournament', err);
      });
  }, [league.league_id]);

  if (!hasTournaments) {
    return null;
  }

  if (hasTournaments && !tournaments.length) {
    return <NoTournament />;
  }

  return (
    <Column style={{ margin: '10px 0' }}>
      {tournaments.map(tournament => (
        <TournamentListItem key={tournament.tournament_id} tournament={tournament} />
      ))}
    </Column>
  );
}

function TournamentPage({ league, actingUser, actingTeam, upcomingTournament }) {
  const activeTournamentId = useMemo(() => league && league.active_tournament_id, [league]);

  const renderNothing = !activeTournamentId && upcomingTournament === 'fetching';

  if (renderNothing) {
    return null;
  }

  return (
    <>
      <NavBar description="Tournament" />
      <Router>
        <SingleTournament
          path=":tournament_id/*"
          league={league}
          actingUser={actingUser}
          actingTeam={actingTeam}
        />
        <TournamentList default league={league} />
      </Router>
    </>
  );
}

export default memo(TournamentPage);
