import React, { Fragment, memo, useMemo, useState } from 'react';

import { Redirect } from '@reach/router';

import { FullSizeLoader } from 'bebo-ui';

import MatchController from 'controllers/match';

// import DownloadPrompt from 'components/download-prompt';
// import Sheet from 'components/modal-sheet';

import useEntity from 'hooks/useEntity';
import useSubscribeTo from 'hooks/useSubscribeTo';
import useActingUser from 'hooks/useActingUser';
import useTrack from 'hooks/useTrack';

import MatchComponent from 'components/match';
import LobbyComponent from 'components/lobby';

import css from './division-match.module.scss';

import useLoadTimes from 'hooks/useLoadTimes';

// import UserController from 'controllers/user';

const findMatchTeam = (match, user_id) => {
  let team = {};

  match.sets.forEach(set => {
    if (
      set &&
      set.team &&
      set.team.users &&
      set.team.users.findIndex(u => u.user_id === user_id) > -1
    ) {
      team = set.team;
    }
  });

  return team;
};

const MatchRoute = memo(({ league, match_id }) => {
  useLoadTimes();
  const league_id = league.league_id;

  const actingUser = useActingUser();
  const [back, setBack] = useState(false);

  const [matchError] = useSubscribeTo('/match', { match_id });
  let gameMatch = useEntity('match', match_id);

  if (!gameMatch) {
    gameMatch = {};
  }

  if (!gameMatch.sets) {
    gameMatch.sets = [];
  }

  const isOwnActiveMatch = useMemo(() => {
    return actingUser.active_match_id === gameMatch.match_id;
  }, [actingUser.active_match_id, gameMatch.match_id]);

  const matchTeam = useMemo(() => findMatchTeam(gameMatch, actingUser.user_id), [
    actingUser.user_id,
    gameMatch
  ]);

  const actingTeam = matchTeam;

  const isOwnMatch = useMemo(() => {
    return matchTeam && matchTeam.team_id ? true : false;
  }, [matchTeam]);

  const teamLeagueId = (
    (gameMatch &&
      ((gameMatch.sets[0] && gameMatch.sets[0].team) ||
        (gameMatch.sets[1] && gameMatch.sets[1].team))) ||
    {}
  ).league_id;

  const redirectUrl = useMemo(() => {
    let l_id = teamLeagueId || league_id;
    let tournament_id = gameMatch.tournament_id;

    if (l_id && tournament_id) {
      return `/app/league/${l_id}/tournament/${tournament_id}`;
    }

    return `/app/league/${l_id}`;
  }, [teamLeagueId, league_id, gameMatch.tournament_id]);

  const track = useTrack('match', match_id, { league_id, team_id: matchTeam.team_id });

  // const { has_pc, has_xbox } = actingUser;

  // const has_xbox_and_pc = useMemo(() => has_pc && has_xbox, [has_pc, has_xbox]);

  let [pickedPlatform] = useState('pc');

  // const pickPlatform = useCallback(current_platform => {
  //   return UserController.update({ current_platform })
  //     .then(() => {
  //       setPickedPlatform(current_platform);
  //     })
  //     .catch(err => {
  //       console.error('failed to pickPlatform', err);
  //     });
  // });

  // console.info('pickedPlatform', pickedPlatform);

  const [sets, ownSet] = useMemo(() => {
    if (!isOwnMatch) {
      return [[...gameMatch.sets], null];
    }
    const ownSetIndex = gameMatch.sets.findIndex(
      set => set && set.team && set.team.team_id === matchTeam.team_id
    );

    let sets = [];
    let ownSet = null;

    if (ownSetIndex === 1) {
      sets = [gameMatch.sets[1], gameMatch.sets[0]];
      ownSet = gameMatch.sets[1];
    } else if (gameMatch.sets.length === 2) {
      if (ownSetIndex === 1) {
        sets = [gameMatch.sets[1], gameMatch.sets[0]];
        ownSet = gameMatch.sets[1];
      } else {
        sets = [gameMatch.sets[0], gameMatch.sets[1]];
        ownSet = gameMatch.sets[0];
      }
    } else {
      sets = [gameMatch.sets[0], null];
      ownSet = gameMatch.sets[0];
    }
    return [sets, ownSet];
  }, [gameMatch, isOwnMatch, matchTeam]);

  const userStates = (gameMatch && gameMatch.data && gameMatch.data.state) || {};

  const ownUserState = useMemo(() => {
    if (!isOwnActiveMatch) {
      return {};
    }
    return userStates[actingUser.user_id] || {};
  }, [isOwnActiveMatch, userStates, actingUser.user_id]);

  const [branReady, pcReady, mixerReady, walleReady, userReady] = useMemo(() => {
    if (!isOwnActiveMatch) {
      return [false, false, false, false, false];
    }

    return [
      ownUserState.bran,
      ownUserState.pc,
      ownUserState.mixer,
      ownUserState.walle,
      ownUserState.ready
    ];
  }, [isOwnActiveMatch, ownUserState]);

  const currentRound = useMemo(() => {
    if (gameMatch.isWaiting) {
      return -1;
    }
    if (isOwnMatch) {
      return ownSet.round;
    }
    return gameMatch.round;
  }, [gameMatch, isOwnMatch, ownSet]);

  const matchState = useMemo(() => {
    if (gameMatch.isWaiting) {
      return 'waiting';
    }
    if (isOwnMatch) {
      return ownSet && ownSet.state;
    }
    return gameMatch && gameMatch.state;
  }, [gameMatch, isOwnMatch, ownSet]);

  const canCancel = Boolean(
    !matchState ? null : matchState === 'waiting' || matchState === 'created'
  );

  const canBack = canCancel && Boolean(gameMatch && gameMatch.tournament_id);

  const showSheet = useMemo(() => {
    if (
      !isOwnActiveMatch ||
      pcReady === undefined ||
      branReady === undefined ||
      userReady === undefined ||
      mixerReady === undefined ||
      walleReady === undefined ||
      canCancel === null ||
      !ownSet ||
      (ownSet && ownSet.state !== 'waiting' && ownSet.state !== 'created')
    ) {
      return null;
    }

    if (!userReady) {
      // if (!pickedPlatform) {
      //   return 'platform';
      // }

      return 'readyUp';
    }

    if (pickedPlatform === 'pc' && !pcReady) {
      return 'pcNoApp';
    }

    if (pickedPlatform === 'xbox') {
      if (!actingUser.mixer_username) {
        return 'mixerUsername';
      }
      if (!mixerReady) {
        return 'mixerStream';
      }
    } else if (pickedPlatform === 'pc') {
      if (!pcReady) {
        return 'pcNoApp';
      }
      if (!walleReady) {
        return 'pcStream';
      }
    }

    if (!branReady) {
      return 'aiDetect';
    }

    return true;
  }, [
    pickedPlatform,
    actingUser.mixer_username,
    isOwnActiveMatch,
    userReady,
    branReady,
    pcReady,
    mixerReady,
    walleReady,
    canCancel,
    ownSet
  ]);

  const isTournamentMatch = useMemo(() => gameMatch && gameMatch.tournament_id, [gameMatch]);

  console.log('**** user states', {
    ownUserState,
    branReady,
    pcReady,
    mixerReady,
    walleReady,
    userReady,
    showSheet
  });

  if (teamLeagueId && league_id && match_id && teamLeagueId !== league_id) {
    return <Redirect noThrow to={`/app/league/${teamLeagueId}/match/${match_id}`} />;
  }

  if (matchError || back) {
    return <Redirect noThrow to={redirectUrl} />;
  }

  if (!(gameMatch && gameMatch.match_id === match_id) || gameMatch.isWaiting) {
    return <FullSizeLoader />;
  }

  const onQuitMatch = () => {
    let prom = Promise.resolve();
    if (canBack) {
      prom = Promise.resolve();
    } else if (canCancel) {
      //cancel
      track('cancel');
      prom = MatchController.cancel(match_id);
    } else if (matchState === 'ended') {
      //ordinary leave
      track('leave');
      prom = Promise.resolve();
    } else {
      //forfeit
      /**
       * text via tyler
       *
       *
       */
      let c = window.confirm(
        'WARNING: This will submit your current score as your final score, and end your match early.\n\nIf you are experiencing technical issues, let us know in Discord first.\n\nAre you sure you want to end your match?'
      );
      if (!c) {
        prom = Promise.reject(null);
      } else {
        track('forfeit');
        prom = MatchController.forfeit(match_id);
      }
    }
    return prom
      .then(() => {
        //this means we can leave, yay
        setBack(true);
      })
      .catch(err => {
        if (err) {
          console.error('failed to leave match', err);
        }
      });
  };

  console.log('*** gameMatch', { gameMatch });

  const onReady = () => {
    return MatchController.readyUp(actingUser.user_id, gameMatch.match_id).catch(err => {
      console.error('failed to ready up', err);
    });
  };

  const onUnReady = () => {
    return MatchController.unReadyUp(actingUser.user_id, gameMatch.match_id).catch(err => {
      console.error('failed to un-ready up', err);
    });
  };

  let renderBody = true;
  if (
    !gameMatch ||
    !(sets[0] && sets[0].state) ||
    (isOwnActiveMatch &&
      ((sets && sets[0] && sets[0].state === 'waiting') ||
        (sets && sets[0] && sets[0].state === 'created')) &&
      !showSheet)
  ) {
    renderBody = false;
  }

  const showLobby =
    isOwnActiveMatch &&
    ((sets && sets[0] && sets[0].state === 'waiting') ||
      (sets && sets[0] && sets[0].state === 'created'));

  return (
    <Fragment>
      {gameMatch && gameMatch.deleted_dttm && <Redirect noThrow to={redirectUrl} />}
      {renderBody && (
        <div className={css.container}>
          {showLobby ? (
            <LobbyComponent
              lobbyState={showSheet}
              league={league}
              onQuitMatch={onQuitMatch}
              redirectUrl={redirectUrl}
              ownUserState={ownUserState}
              userStates={userStates}
              matchRounds={gameMatch.game_cnt}
              canCancel={canCancel}
              canBack={canBack}
              actingTeam={actingTeam}
              track={track}
              onReady={onReady}
              onUnReady={onUnReady}
              isTournamentMatch={isTournamentMatch}
            />
          ) : (
            <MatchComponent
              league={league}
              onQuitMatch={onQuitMatch}
              redirectUrl={redirectUrl}
              sets={sets}
              isOwnMatch={isOwnMatch}
              currentRound={currentRound}
              isActiveMatch={isOwnActiveMatch}
              ownUserState={ownUserState}
              userStates={userStates}
              canCancel={canCancel}
              canBack={canBack}
              matchRounds={gameMatch.game_cnt}
              actingTeam={actingTeam}
              track={track}
              isTournamentMatch={isTournamentMatch}
              matchState={gameMatch.state}
              matchOutcome={gameMatch.outcome}
            />
          )}
        </div>
      )}
    </Fragment>
  );
});

export default MatchRoute;
