import React, { useEffect, useMemo } from 'react';

import OverlayUI from './ui';
import ErrorBoundary from 'components/ErrorBoundary';
import useActingUser from 'hooks/useActingUser';
import useSubscribeTo from 'hooks/useSubscribeTo';
import useEntity from 'hooks/useEntity';
import useSocketSubscription from 'hooks/useSocketSubscription';

import { subscribe, unsubscribe } from 'classes/socket';

// const Overlay = () => {
//   useEffect(() => {
//     const bodyElem = document.getElementsByTagName('body')[0];
//     const prevBackground = bodyElem.style.background;
//     bodyElem.style.background = null;

//     return () => {
//       bodyElem.style.background = prevBackground;
//     };
//   }, []);

//   return <OverlayUI />;
// };

// export default Overlay;

const Overlay = () => {
  const actingUser = useActingUser();
  const [activeTournamentId, actingTeam] = useMemo(() => {
    if (!actingUser) {
      return [null, null];
    }
    let tournament_id = null;
    let actingTeam = null;
    for (let league of actingUser.leagues) {
      if (league.active_tournament_id) {
        tournament_id = league.active_tournament_id;
        actingTeam = league.teams && league.teams[0] ? league.teams[0] : null;
        break;
      }
    }
    return [tournament_id, actingTeam];
  }, [actingUser]);

  console.log('*** actingUser', actingUser, activeTournamentId, actingTeam);

  if (!actingUser || !activeTournamentId || !actingTeam) {
    return null;
  }

  return (
    <TournamentContainer
      actingUser={actingUser}
      actingTeam={actingTeam}
      tournament_id={activeTournamentId}
    />
  );
};

const TournamentContainer = ({ actingUser, tournament_id, actingTeam }) => {
  useSubscribeTo('/tournament', { tournament_id });
  const tournament = useEntity('tournament', tournament_id);

  const tournamentRound = parseInt((tournament && tournament.round) || 0, 10);

  const [unfilteredTeams] = useSocketSubscription(
    '/tournament/team',
    { tournament_id },
    undefined,
    {
      initOptions: { count: 500 }
    }
  );

  const [initialRoundMatches] = useSocketSubscription(
    '/tournament/match',
    {
      tournament_id,
      tournament_round: tournamentRound
    },
    undefined,
    {
      initOptions: { count: 500 },
      pollIf: result => (!result || (Array.isArray(result) && !result.length) ? 20 * 1000 : false)
    }
  );

  const hasRoundMatches = useMemo(() => Boolean(initialRoundMatches), [initialRoundMatches]);

  const entityMatches = useEntity(
    'match',
    (initialRoundMatches || []).map(match => match.match_id)
  );

  // const tournamentState = tournament && tournament.state;

  const teams = Array.isArray(unfilteredTeams)
    ? unfilteredTeams.filter(team => Boolean(team.checkin_dttm))
    : unfilteredTeams;

  const roundMatches = useMemo(() => {
    if (!entityMatches || !teams) {
      return [];
    }
    for (let match of entityMatches) {
      for (let set of match.sets) {
        set._seed = (teams.find(t => t.team_id === set.team_id) || {}).seed || '?';
      }
      match.sets = match.sets.sort((a, b) => a._seed - b._seed);
    }

    return entityMatches;
  }, [entityMatches, teams]).filter(m => m && m.tournament_round === tournamentRound);

  useEffect(() => {
    let matches = roundMatches || [];
    //subscribe to all non-ended matches
    let subscribedMatchIds = [];
    for (let match of matches) {
      if (match && match.match_id && match.state && match.state !== 'ended') {
        subscribe('/match', { match_id: match.match_id }).catch(err =>
          console.error('failed to subscribe to match', err)
        );

        subscribedMatchIds.push(match.match_id);
      }
    }

    return () => {
      for (let match_id of subscribedMatchIds) {
        unsubscribe('/match', { match_id }).catch(err =>
          console.error('failed to unsubscribe from match', err)
        );
      }
    };
  }, [roundMatches]);

  const hasOwnTeamInTournament = useMemo(() => {
    if (!actingTeam || !actingTeam.team_id || !teams || !teams.length) {
      return false;
    }
    return teams.findIndex(t => t.team_id === actingTeam.team_id) > -1;
  }, [actingTeam, teams]);

  const [ownNextTournamentMatch] = useMemo(() => {
    if (!hasRoundMatches || !hasOwnTeamInTournament || (!actingTeam && actingTeam.team_id)) {
      return [null, null];
    }
    let match = null;
    let set = null;

    for (let m of roundMatches) {
      for (let s of m.sets) {
        if (s.team.team_id === actingTeam.team_id) {
          match = m;
          set = s;
          if (s && m) {
            s._match_id = m.match_id;
          }
          break;
        }
      }
    }
    return [match, set];
  }, [hasRoundMatches, roundMatches, hasOwnTeamInTournament, actingTeam]);
  console.log('*** ', { tournament, ownNextTournamentMatch });

  if (!tournament || !hasOwnTeamInTournament) {
    return null;
  }

  return (
    <OverlayUI tournament={tournament} match={ownNextTournamentMatch} actingTeam={actingTeam} />
  );
};

const Wrapper = () => {
  useEffect(() => {
    const bodyElem = document.getElementsByTagName('body')[0];
    const prevBackground = bodyElem.style.background;
    bodyElem.style.background = null;
    return () => {
      bodyElem.style.background = prevBackground;
    };
  }, []);

  return (
    <ErrorBoundary isSilent>
      <Overlay />
    </ErrorBoundary>
  );
};

export default Wrapper;
