import React, { memo, useMemo, useState, useEffect } from 'react';

import { FullSizeLoader, Loader } from 'bebo-ui';

import { isNumber } from 'scripts/utils';

import useSocketSubscription from 'hooks/useSocketSubscription';

import useLoadTimes from 'hooks/useLoadTimes';

import useEntity from 'hooks/useEntity';

// import NavBar from 'components/nav-bar';

import css from './tournament.module.scss';

import MatchCard from 'components/match-card';

import Header from 'components/tournament/header';
import ProgressBar from 'components/tournament/progress-bar';
import TeamCard from 'components/tournament/team-card';
import FloatingCard from 'components/tournament/card';

import { subscribe, unsubscribe } from 'classes/socket';

const TournamentContainer = memo(({ league, tournament, actingTeam, actingUser, singleElim }) => {
  useLoadTimes();

  const tournament_id = tournament && tournament.tournament_id;

  const [userSelectedRound, setUserSelectedRound] = useState(null);

  const tournamentRound = useMemo(() => {
    if (!tournament || !tournament.round) {
      return 0;
    }
    return parseInt(tournament.round);
  }, [tournament]);

  const selectedRound = useMemo(() => {
    if (isNumber(userSelectedRound)) {
      return userSelectedRound;
    }
    return tournamentRound;
  }, [tournamentRound, userSelectedRound]);

  const tournamentState = useMemo(() => {
    if (!tournament) {
      return null;
    }
    return tournament.state;
  }, [tournament]);

  const [unfilteredTeams] = useSocketSubscription(
    '/tournament/team',
    { tournament_id },
    undefined,
    {
      initOptions: { count: 500 }
    }
  );

  const teams = Array.isArray(unfilteredTeams)
    ? unfilteredTeams.filter(team => Boolean(team.checkin_dttm))
    : unfilteredTeams;

  const [initialRoundMatches] = useSocketSubscription(
    '/tournament/bracket',
    {
      tournament_id,
      tournament_round: selectedRound
    },
    undefined,
    {
      initOptions: { count: 500 },
      pollIf: result => (!result || (Array.isArray(result) && !result.length) ? 20 * 1000 : false)
    }
  );

  const hasRoundMatches = useMemo(() => Boolean(initialRoundMatches), [initialRoundMatches]);

  const entityMatches = useEntity(
    'match',
    (initialRoundMatches || []).map(match => match.match_id)
  );

  const roundMatches = useMemo(() => {
    if (!entityMatches || !teams) {
      return [];
    }
    for (let match of entityMatches) {
      for (let set of match.sets) {
        set._seed = (teams.find(t => t.team_id === set.team_id) || {}).seed || '?';
      }
      match.sets = match.sets.sort((a, b) => a._seed - b._seed);
    }

    return entityMatches;
  }, [entityMatches, teams]).filter(m => m && m.tournament_round === selectedRound);

  useEffect(() => {
    let matches = roundMatches || [];
    //subscribe to all non-ended matches
    let subscribedMatchIds = [];
    for (let match of matches) {
      if (match && match.match_id && match.state && match.state !== 'ended') {
        subscribe('/match', { match_id: match.match_id }).catch(err =>
          console.error('failed to subscribe to match', err)
        );

        subscribedMatchIds.push(match.match_id);
      }
    }

    return () => {
      for (let match_id of subscribedMatchIds) {
        unsubscribe('/match', { match_id }).catch(err =>
          console.error('failed to unsubscribe from match', err)
        );
      }
    };
  }, [roundMatches]);

  const hasOwnTeamInTournament = useMemo(() => {
    if (!actingTeam || !actingTeam.team_id || !teams || !teams.length) {
      return false;
    }
    return teams.findIndex(t => t.team_id === actingTeam.team_id) > -1;
  }, [actingTeam, teams]);

  const [ownNextTournamentMatch, ownSet] = useMemo(() => {
    if (!hasRoundMatches || !hasOwnTeamInTournament || (!actingTeam && actingTeam.team_id)) {
      return [null, null];
    }
    let match = null;
    let set = null;

    for (let m of roundMatches) {
      for (let s of m.sets) {
        if (s.team.team_id === actingTeam.team_id) {
          match = m;
          set = s;
          if (s && m) {
            s._match_id = m.match_id;
          }
          break;
        }
      }
    }
    return [match, set];
  }, [hasRoundMatches, roundMatches, hasOwnTeamInTournament, actingTeam]);

  const tournamentRoundText = useMemo(() => {
    if (!tournament) {
      return null;
    }
    if (tournamentState === 'started') {
      return `Round ${tournamentRound}`;
    }
    return tournamentState;
  }, [tournament, tournamentRound, tournamentState]);

  const tournamentRounds = useMemo(() => {
    if (!tournament || !tournament.round_cnt || tournament.round_cnt < 0) {
      return [];
    }

    let rounds = [];
    for (let i = 0; i < tournament.round_cnt; i++) {
      rounds[i] = { index: i + 1, isFinal: i === tournament.round_cnt - 1 };
    }

    return rounds;
  }, [tournament]);

  const isFinal = useMemo(() => {
    return (
      tournamentRounds &&
      tournamentRound &&
      tournamentRounds[selectedRound - 1] &&
      tournamentRounds[selectedRound - 1].isFinal
    );
  }, [tournamentRounds, selectedRound, tournamentRound]);

  const isUpperBracket = useMemo(() => {
    if (!teams || !hasRoundMatches || !roundMatches.length || !tournamentState) {
      return false;
    }
    if (
      hasOwnTeamInTournament &&
      ((ownNextTournamentMatch && ownNextTournamentMatch.tournament_bracket === 'upper') ||
        !ownNextTournamentMatch)
    ) {
      return true;
    }
    return false;
  }, [
    hasRoundMatches,
    hasOwnTeamInTournament,
    tournamentState,
    teams,
    roundMatches.length,
    ownNextTournamentMatch
  ]);

  const hasAdvanced = useMemo(() => {
    if (
      !teams ||
      !hasRoundMatches ||
      !roundMatches.length ||
      !tournamentState ||
      !ownNextTournamentMatch ||
      !ownSet
    ) {
      return false;
    }

    if (
      hasOwnTeamInTournament &&
      ((ownSet && ownSet.outcome === 'win') ||
        (ownSet &&
          ownSet.outcome === 'bye' &&
          !(ownNextTournamentMatch.tournament_bracket === 'lower' && isFinal)))
    ) {
      return true;
    }
    return false;
  }, [
    hasRoundMatches,
    hasOwnTeamInTournament,
    tournamentState,
    ownNextTournamentMatch,
    isFinal,
    teams,
    roundMatches.length,
    ownSet
  ]);

  const hasTied = useMemo(() => {
    if (
      !teams ||
      !hasRoundMatches ||
      !roundMatches.length ||
      !tournamentState ||
      !ownNextTournamentMatch ||
      !ownSet
    ) {
      return false;
    }

    if (hasOwnTeamInTournament && (ownSet && ownSet.outcome === 'draw')) {
      return true;
    }
    return false;
  }, [
    hasRoundMatches,
    hasOwnTeamInTournament,
    tournamentState,
    ownNextTournamentMatch,
    teams,
    roundMatches.length,
    ownSet
  ]);

  const cardType = useMemo(() => {
    if (!hasOwnTeamInTournament) {
      return null;
    }
    if (
      !tournamentState ||
      (tournamentState !== 'started' && tournamentState !== 'ended') ||
      !hasRoundMatches ||
      !roundMatches.length
    ) {
      return null;
    }
    if (
      ownNextTournamentMatch &&
      ownNextTournamentMatch.state !== 'ended' &&
      ownNextTournamentMatch.state !== 'bye' &&
      tournamentState !== 'ended'
    ) {
      return 'next_match';
    }

    if (hasAdvanced && isFinal) {
      return 'tournament_winner';
    } else if (hasAdvanced) {
      return 'round_advanced';
    } else if (hasTied) {
      return 'draw';
    } else {
      if (isUpperBracket && !isFinal) {
        return 'lower_bracket';
      }
      return 'round_eliminated';
    }
  }, [
    hasTied,
    isFinal,
    hasRoundMatches,
    tournamentState,
    hasOwnTeamInTournament,
    ownNextTournamentMatch,
    hasAdvanced,
    roundMatches,
    isUpperBracket
  ]);

  if (!tournament || tournament.isWaiting || !actingUser || !tournamentState) {
    return <FullSizeLoader fixed />;
  }

  const upperBracket =
    roundMatches && roundMatches.length
      ? roundMatches.filter(match => match.tournament_bracket === 'upper')
      : [];
  const lowerBracket =
    roundMatches && roundMatches.length
      ? roundMatches.filter(match => match.tournament_bracket === 'lower')
      : [];

  return (
    <div className={css.container}>
      {/* <NavBar /> */}

      <Header tournamentRoundText={tournamentRoundText}>
        <ProgressBar
          currentRound={tournamentRound}
          selectedRound={selectedRound}
          rounds={tournamentRounds}
          setUserSelectedRound={setUserSelectedRound}
        />
      </Header>

      <FloatingCard
        cardType={cardType}
        ownSet={ownSet}
        actingUser={actingUser}
        league={league}
        selectedRound={selectedRound}
        actingTeam={actingTeam}
        ownNextTournamentMatch={ownNextTournamentMatch}
        singleElim={singleElim}
      />

      <>
        {tournamentState === 'created' && teams ? (
          <div className={css.scrollBodyWrapper}>
            <h4>Qualified Teams</h4>
            {teams.map((team, i) => (
              <TeamCard key={team.team_id} team={team} />
            ))}
          </div>
        ) : roundMatches && roundMatches.length ? (
          <>
            {upperBracket.length ? (
              <Bracket
                header={singleElim ? '' : `Upper Bracket${isFinal ? ' Final' : ''}`}
                subHeader={`(${upperBracket[0].game_cnt} Game${
                  upperBracket[0].game_cnt > 1 ? 's' : ''
                })`}
                bracket={upperBracket}
                league_id={league.league_id}
                tournamentRounds={tournamentRounds}
                selectedRound={selectedRound}
              />
            ) : null}
            {!singleElim && lowerBracket.length ? (
              <Bracket
                header={`Lower Bracket${
                  tournamentRounds.length - 1 === selectedRound ? ' Final' : ''
                }`}
                subHeader={`(${lowerBracket[0].game_cnt} Game${
                  lowerBracket[0].game_cnt > 1 ? 's' : ''
                })`}
                bracket={lowerBracket}
                league_id={league.league_id}
                selectedRound={selectedRound}
                tournamentRounds={tournamentRounds}
              />
            ) : null}
          </>
        ) : (
          <div
            style={{
              position: 'absolute',
              left: '50%',
              top: '33%',
              transform: 'translate(-50%, -50%)',
              display: 'flex',
              flexDirection: 'column'
            }}
          >
            <Loader style={{ margin: '10px auto' }} size={25} />
            <p>
              {selectedRound === tournamentRound ? 'Generating Bracket...' : 'Loading Round...'}
            </p>
          </div>
        )}
      </>
    </div>
  );
});

export default TournamentContainer;

const Bracket = memo(
  ({ header, subHeader, bracket, league_id, selectedRound, tournamentRounds }) => {
    return (
      <div className={css.bracket}>
        <span className={css.bracketHeader}>{header}</span>
        {subHeader && <span className={css.bracketSubHeader}>{subHeader}</span>}
        <ul className={css.bracketMatches}>
          {bracket.map((match, i) => (
            <MatchCard
              link={`/app/league/${league_id}/match/${match.match_id}`}
              key={match.match_id ? match.match_id : `match_${i}`}
              match={match}
              index={i}
              round={tournamentRounds[selectedRound - 1] || {}}
            />
          ))}
        </ul>
      </div>
    );
  }
);
