import store from 'store2';
import { getStatusText } from 'http-status-codes';

export const storage = store.namespace('bebo');
export const sessionStorage = store.namespace('bebo_session');

//TODO consider if this needs to live remotely (probably does)
export const getUserStorage = user_id => store.namespace(`bebo_user_${user_id}`);

export const uuid = () => {
  function _p8(s) {
    const p = (Math.random().toString(16) + '000000000').substr(2, 8);
    return s ? '-' + p.substr(0, 4) + '-' + p.substr(4, 4) : p;
  }

  return _p8() + _p8(true) + _p8(true) + _p8();
};

export const getParameterByName = (name, url) => {
  if (!url) url = window.location.href;
  name = name.replace(/[[\]]/g, '\\$&');
  var regex = new RegExp('[?&]' + name + '(=([^&#]*)|&|#|$)'),
    results = regex.exec(url);
  if (!results) return null;
  if (!results[2]) return '';
  return decodeURIComponent(results[2].replace(/\+/g, ' '));
};

export const getSingleResult = message => (message.result && message.result[0]) || null;

export const isDesktopApp = () =>
  (/^bebo/i.test(navigator.userAgent) &&
    window.location.toString().indexOf('/close') === -1 &&
    window.location.toString().indexOf('/oauth') === -1 &&
    window.location.toString().indexOf('/legal') === -1) ||
  window.location.toString().indexOf('/overlay') > -1;

export const isNumber = num => {
  if (typeof num === 'number') {
    return num - num === 0;
  }
  if (typeof num === 'string' && num.trim() !== '') {
    return Number.isFinite ? Number.isFinite(+num) : isFinite(+num);
  }
  return false;
};

export const getRoutingKey = (url, payload, isSingleValue) => {
  url = isSingleValue ? url : `list.${url}`;
  if (!payload) {
    return url;
  }
  const routingKey = `${url}.${Object.keys(payload)
    .sort()
    .map(key => `${key}:${payload[key]}`)
    .join('.')}`;

  return routingKey;
};

export const reverseRoutingKey = routing_key => {
  let isSingleValue = false;
  let url = null;
  const payload = {};
  const split = routing_key.split('.');
  let [l, u, ...pl] = split;

  if (l === 'list') {
    isSingleValue = false;
    url = u;
  } else {
    isSingleValue = true;
    url = l;
    pl = [u, ...pl];
  }

  pl.filter(pair => pair).forEach(pair => {
    let [key, value] = pair.split(':');
    payload[key] = value;
  });
  return { isSingleValue, url, payload };
};

export const isPWA = () =>
  window.matchMedia('(display-mode: standalone)').matches || window.navigator.standalone;

export const getVersionFromString = versionString => {
  const [major, minor, patch] = versionString.split('.');
  return { major, minor, patch };
};

export const getBeboAppVersion = () => {
  if (!isDesktopApp()) {
    return null;
  }
  try {
    const versionString = navigator.userAgent.match(/(\d+\.\d+\.\d+)/i)[0]; //no /g because we only want the first occurrence
    const { major, minor, patch } = getVersionFromString(versionString);
    return { full: versionString, major, minor, patch };
  } catch (err) {
    console.error('failed to getBeboAppVersion', err);
    return null;
  }
};

export const isAppVersionBigger = minVersion => {
  if (!isDesktopApp()) {
    return true;
  }
  const currentAppVersion = getBeboAppVersion();
  const minimumAppVersion = getVersionFromString(minVersion);
  if (currentAppVersion.major > minimumAppVersion.major) {
    return true;
  }
  if (currentAppVersion.minor > minimumAppVersion.minor) {
    return true;
  }
  if (currentAppVersion.patch > minimumAppVersion.patch) {
    return true;
  }
  return false;
};

export const hasErrorShape = params => {
  return params.hasOwnProperty('code') && params.hasOwnProperty('ack_id');
};

export const getMessageFromCode = code => {
  return `Whoops, something went wrong.\n(Code ${code}: ${getStatusText(code)})`;
};

export const transformErrorShape = ({ code, message, ...rest }) => {
  return {
    ...rest,
    message: message && parseInt(message, 10) !== code ? message : getMessageFromCode(code)
  };
};

export const isDefaultImage = image_url => {
  if (!image_url || !image_url.includes('a.imgdropt')) {
    return true;
  }
  return false;
};
