import { BehaviorSubject } from 'rxjs';
import { isAfter, isEqual } from 'date-fns';

const GLOBAL_ENTITY_STORE = new Map();

export const handleIncomingMessage = message => {
  //check if message has a result key

  if (!message.result || !Array.isArray(message.result) || !message.result.length) {
    return;
  }
  //loop over result objects
  message.result.forEach(obj => {
    //if obj has a URL key on it treat it as an entity otherwise we can ignore it
    if (!obj.url) {
      return;
    }
    const type = obj.url.replace('/', '');
    const id = obj[`${type}_id`];
    upsertEntity(type, id, obj);
  });
};

export const getEntity = (type, id) => {
  const entity_store = getEntityStore(type);

  if (entity_store.has(id)) {
    return entity_store.get(id);
  }
  return upsertEntity(type, id, {
    updated_dttm: new Date(0),
    isWaiting: true,
    url: `/${type}`,
    [`${type}_id`]: id
  });
};

const getEntityStore = type => {
  if (GLOBAL_ENTITY_STORE.has(type)) {
    return GLOBAL_ENTITY_STORE.get(type);
  }
  const ENTITY_STORE = new Map();
  GLOBAL_ENTITY_STORE.set(type, ENTITY_STORE);
  return ENTITY_STORE;
};

const upsertEntity = (type, id, entity) => {
  if (!type || !id || !entity) {
    return;
  }

  const ENTITY_STORE = getEntityStore(type);

  //check if the entity already exists
  if (!ENTITY_STORE.has(id)) {
    //if not then we can simply insert
    const new_entity_subject = new BehaviorSubject(entity);
    ENTITY_STORE.set(id, new_entity_subject);
    return new_entity_subject;
  }

  //check if the new updated_dttm is > than the stored updated_dttm;
  const existing_entity_subject = ENTITY_STORE.get(id);
  const existing_entity = existing_entity_subject.getValue();
  if (
    isAfter(entity.updated_dttm, existing_entity.updated_dttm) ||
    isEqual(entity.updated_dttm, existing_entity.updated_dttm)
  ) {
    //if it is we need to update & insert
    //otherwise we ignore
    existing_entity_subject.next({ ...existing_entity, ...entity, isWaiting: false });
  }

  return existing_entity_subject;
};
