import { onmessage$, GET } from 'classes/socket';
import { filter, map } from 'rxjs/operators';
import { BehaviorSubject, from, merge } from 'rxjs';
import invariant from 'invariant';

import { getRoutingKey } from 'scripts/utils';

const subjectSubscriptions = new Map();

const filterFn = (message, filters) => {
  invariant(message, 'filterFN requires a message to be passed to it');
  invariant(filters, 'filterFN requires filters to be passed to it');

  const filterKeys = Object.keys(filters);
  let filter = true;
  for (let key of filterKeys) {
    if (message[key] !== filters[key]) {
      filter = false;
      break;
    }
  }

  return filter;
};

export const getSubject = (
  url,
  filters,
  initialValue,
  isSingleValue,
  noInit,
  reFetch,
  initOptions,
  pollInterval
) => {
  invariant(
    url,
    `getSubject requires at least a "url" to be passedas the first parameter, but got: ${url}`
  );
  const routingKey = getRoutingKey(url, filters, isSingleValue);

  //if we already have a subject we will just use it
  if (subjectSubscriptions.has(routingKey)) {
    const s = subjectSubscriptions.get(routingKey).subject;
    if (reFetch) {
      from(GET(url, { ...filters, ...initOptions }))
        .pipe(
          map(message => (isSingleValue ? message.result && message.result[0] : message.result))
        )
        .subscribe(message => s.next(message), console.error);
    }
    return s;
  }

  //if we do not already have a subject we will create a new one and hook it up
  const subject = new BehaviorSubject(initialValue);

  filters = { ...filters, url };

  //initialize with a GET
  const subscription = (noInit
    ? onmessage$.pipe(filter(message => filterFn(message, filters)))
    : merge(
        from(GET(url, { ...filters, ...initOptions })),
        onmessage$.pipe(filter(message => filterFn(message, filters)))
      )
  )
    .pipe(map(message => (isSingleValue ? message.result && message.result[0] : message.result)))
    .subscribe(subject);

  //add it to the subjectSubscriptions
  subjectSubscriptions.set(routingKey, { subject, subscription });

  return subject;
};

export const removeSubject = (url, filters, isSingleValue) => {
  invariant(
    url,
    `removeSubject requires at least a "url" to be passedas the first parameter, but got: ${url}`
  );
  const routingKey = getRoutingKey(url, filters, isSingleValue);
  if (subjectSubscriptions.has(routingKey)) {
    //first we get the actual subject
    const { subject, subscription } = subjectSubscriptions.get(routingKey);
    //now we need to check if there are no more active subscriptions to this subject
    if (!subject.observers.length) {
      //if not we can safely unsubscribe
      subscription.unsubscribe();
      //and then remove the entire thing from the subjectSubscriptions set
      subjectSubscriptions.delete(routingKey);
    }
  } else {
    console.warn('trying to removeSubject but it already does not exist anymore', {
      routingKey
    });
  }
};
