package storage

import (
	"context"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	dynamoDBTimeout = 5 * time.Second
)

type dao struct {
	ddb          dynamodbiface.DynamoDBAPI
	tableNameKey string

	tableName *string
}

//Entry represents an arbitrary DynamoDB entry
type Entry interface{}

// TODO: why return pointer to string?
func (d *dao) getTableName() *string {
	if d.tableName == nil {
		d.tableName = aws.String(os.Getenv(d.tableNameKey))
	}

	return d.tableName
}

func (d *dao) put(ctx context.Context, entry Entry) error {
	av, err := dynamodbattribute.MarshalMap(entry)

	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		Item:      av,
		TableName: d.getTableName(),
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()

	_, err = d.ddb.PutItemWithContext(ctx, input)

	return err
}

func (d *dao) get(ctx context.Context, searchKey map[string]interface{}, output Entry) error {

	key, err := dynamodbattribute.MarshalMap(searchKey)
	if err != nil {
		return err
	}

	input := &dynamodb.GetItemInput{
		TableName:      d.getTableName(),
		ConsistentRead: aws.Bool(true), // TODO: Why?
		Key:            key,
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()

	result, err := d.ddb.GetItemWithContext(ctx, input)
	if err != nil {
		return err
	}

	if result == nil {
		return err
	}

	return dynamodbattribute.UnmarshalMap(result.Item, output)
}

// Delete deletes an entry by ID
func (d *dao) hardDelete(ctx context.Context, searchKey map[string]interface{}, output interface{}) error {
	key, err := dynamodbattribute.MarshalMap(searchKey)
	if err != nil {
		return err
	}

	input := &dynamodb.DeleteItemInput{
		TableName:    d.getTableName(),
		Key:          key,
		ReturnValues: aws.String("ALL_OLD"),
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()

	result, err := d.ddb.DeleteItemWithContext(ctx, input)
	if err != nil {
		return err
	}

	return dynamodbattribute.UnmarshalMap(result.Attributes, output)
}

func (d *dao) query(ctx context.Context, expr string, vals map[string]string, output interface{}) error {
	expressionAttributeValues := map[string]*dynamodb.AttributeValue{}

	for k, v := range vals {
		expressionAttributeValues[k] = &dynamodb.AttributeValue{
			S: aws.String(v),
		}
	}

	input := &dynamodb.QueryInput{
		TableName:                 d.getTableName(),
		KeyConditionExpression:    aws.String(expr),
		ExpressionAttributeValues: expressionAttributeValues,
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()

	result, err := d.ddb.QueryWithContext(ctx, input)
	if err != nil {
		return err
	}

	return dynamodbattribute.UnmarshalListOfMaps(result.Items, output)
}
