package twirp

import (
	"context"
	"fmt"
	"os"
	"time"

	"code.justin.tv/esports-exp/vision/internal/storage"
	"github.com/aws/aws-sdk-go/service/dynamodb"

	serviceapi "code.justin.tv/esports-exp/vision/pkg/client"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/twitchtv/twirp"
)

type VisionServer struct {
	tokenDao *storage.TokenDao
}

func NewVisionServer() (*VisionServer, error) {
	sess, err := session.NewSession(&aws.Config{
		// Current region is pre-defined in the Lambda execution
		// environment
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		return nil, err
	}

	ddb := dynamodb.New(sess)

	if err != nil {
		return nil, err
	}

	tokenDao := storage.NewTokenDao(ddb)
	return &VisionServer{
		tokenDao,
	}, nil
}

func (v *VisionServer) AddToken(ctx context.Context, req *serviceapi.AddTokenRequest) (*serviceapi.AddTokenResponse, error) {
	if req.Token == "" {
		return nil, twirp.RequiredArgumentError("Token")
	}

	if req.Platform == "" {
		return nil, twirp.RequiredArgumentError("Platform")
	}

	if req.Capabilities == nil {
		return nil, twirp.RequiredArgumentError("Capabilities")
	}

	if req.OnSiteCapabilities == nil {
		return nil, twirp.RequiredArgumentError("OnSiteCapabilities")
	}

	entry := storage.TokenEntry{
		DeviceId: req.DeviceId,
		Token:    req.Token,
		Platform: req.Platform,
		Name:     req.Name,
		AppBuild: req.AppBuild,

		Capabilities:       req.Capabilities,
		OnSiteCapabilities: req.OnSiteCapabilities,
		Languages:          req.Languages,

		CreatedDTTM: time.Now().UTC(),
	}

	fmt.Printf("Saving entry: %#v \n", entry)

	err := v.tokenDao.Put(ctx, &entry)

	if err != nil {
		return nil, err
	}

	return &serviceapi.AddTokenResponse{}, nil
}
